import weakref
import os

import pygame

from enum import *

import render
import maps
import content
import input
import combat
import mobs
import font
import ui
import engine
import pathing
import config
import save
import music
import sound
import skills

class Mode:
    DEFAULT = 0
    TARGET = 1
    ATTACK = 2
    SKILLPANE = 3
    POSSESSION = 4
    AI = 5
    STATSPANE = 6
    DEAD = 7
    TITLESCREEN = 8
    MAINMENU = 9
    INGAMEMENU = 10
    OPTIONS = 11
    ENDTURN = 12
    FORMPANE = 13

class Cursor(object):
    def __init__(self):
        self.visible = False
        self._source = (0, 0)
        self.range = 1
        self.x = 0
        self.y = 0
        self.func = None
        self.skill = None
        
    def tryMove(self, x, y, absolute = False):
        if absolute:
            self.x = x
            self.y = y
        else:
            self.x += x
            self.y += y
        
        success = True
        
        if abs(self.x - self._source[0]) > self.range:
            success = False
            if self.x < self._source[0]:
                self.x = self._source[0] - self.range
            else:
                self.x = self._source[0] + self.range
        if abs(self.y - self._source[1]) > self.range:
            success = False
            if self.y < self._source[1]:
                self.y = self._source[1] - self.range
            else:
                self.y = self._source[1] + self.range
                
        return success
                
    def setSource(self, value):
        self._source = value
        self.x = value[0]
        self.y = value[1]

    def getSource(self):
        return self._source
    
class Game(render.Renderable):
    def __init__(self):
        render.Renderable.__init__(self)
        
        self.configFile = 'midboss.ini'
        
        pygame.mixer.pre_init(44100, -16, 4, 2048)
        pygame.init()
        pygame.display.set_icon(pygame.image.load("misc/icon.png"))
        pygame.display.set_caption("MidBoss")
        
        self.volume = (50, 100, 100)
        self.attackOnBump = False
        self.attackConfirms = False
        self.rememberTarget = False
        self.multiKeyDiagonals = False
        
        self.baseDir = None
        self.hasMovedDiagonally = False
        
        self.config = config.Config(self.configFile)
        
        self.quitting = False
        self.input = input.InputHandler()
        self.renderer = render.Renderer(engine.scale)
        content.init()
        
        self.renderer.renderables.append(self)
        self.ui = ui.UI(self.renderer)
        self.ui.options.config = self.config
        
        self.readConfig()
        self.lastTick = 0
        
        channels = pygame.mixer.get_num_channels()
        # TODO: Die if less than 3
        pygame.mixer.set_reserved(2)
        self.music = None
        
        pygame.mixer.music.load("misc/imps lament.ogg")
        pygame.mixer.music.play()
        self.setVolume(self.volume)
        
        self.mode = Mode.TITLESCREEN
        self.cursor = Cursor()
        
        self.saveFile = "save.db"
        self.level = None
        self.levels = []
        self.playersTurn = True
        self.moveSpeed = 50
        self.save = None
        
        self.mobiles = []
        self.moveElapsed = 0
        
        self.target = None
        self.curSound = None
        self.soundElapsed = 0
        self.soundFunc = None
        
        self.activity = None
        
        self.lastTarget = None
        self.lastHP = 0
        
        self.input.pressAnyKey(self.anyKey)
        
    def startGame(self, playerData = None):
        pygame.mixer.music.fadeout(500)
        
        if self.music == None:
            self.music = music.MusicHandler([pygame.mixer.Channel(0), pygame.mixer.Channel(1)])
            self.setVolume(self.volume)
            self.music.addMusic(content.getMusic("dungeon1"))
            self.music.addMusic(content.getMusic("dungeon2"))
            self.music.addMusic(content.getMusic("dungeon3"))

        if self.level != None:
            self.level.cleanUp()
        
        if self.save == None or self.save.closed:
            self.save = save.createNew(self.saveFile)
        mobs.curID = 0
        
        engine.player.reset()
        engine.clearLog()
        self.ui.mainMenu.hide()
        self.ui.bottomPane.show()
        self.ui.textLog.show()
        self.ui.deathScreen.hide()
        self.mode = Mode.DEFAULT
        
        self.levels = []
        
        if playerData == None:
            self.level = maps.generate(engine.roomSize, engine.floorWidth, engine.floorHeight)
            self.save.addFloor(self.level)
        
            engine.player.health = engine.player.mobile.maxHealth
            engine.player.str = engine.player.mobile.str
            engine.player.agi = engine.player.mobile.agi
            engine.player.con = engine.player.mobile.con
            engine.player.mag = engine.player.mobile.mag
            engine.player.wil = engine.player.mobile.wil
            engine.player.mobile.health = engine.player.mobile.maxHealth
            
            engine.player.floor = 1
            engine.player.exploredFloors = 1
                    
            self.levels.append(self.level)
        else:
            exploredFloors = playerData["exploredFloors"]
            curFloor = playerData["floor"]
            
            for i in xrange(1, exploredFloors+1):
                data = self.save.getFloorData(i)
                self.levels.append(maps.load(data))
                
                if i == curFloor:
                    self.level = self.levels[-1]
        
        self.level.setDeathCallback(self.mobileDied)
        
        self.renderer.renderables.insert(0, self.level.renderer)
        
        self.playersTurn = True
        self.moveSpeed = 50
        
        self.mobiles = []
        self.moveElapsed = 0
        
        self.target = None
        self.curSound = None
        self.soundElapsed = 0
        self.soundFunc = None
        
        self.activity = None
        
        self.keyBinds = []
        
        engine.log("Welcome to Mid Boss!")
        engine.log("Press 'L' to open or close this log.")
        engine.log("")
        engine.log("You are an imp, a weak creature with the ability to")
        engine.log("possess other creatures. You want to become the")
        engine.log("dungeon's final boss, who resides on floor 12.")
        engine.log("")
        engine.log("Possess creatures to learn their abilities, level up")
        engine.log("to add stat bonusses to any possessed creature.")
        engine.log("")
        engine.log("Numpad to move, ASD for attack, skill, and status.")
        engine.log("Space/numpad 5 to wait, F9/F10 to resize the window.")
        engine.log("")
        engine.log("")
        engine.log("")
        engine.log("")
        engine.log("")
        
        if playerData != None:
            engine.player.load(playerData)
            engine.player.possess(self.level.mobiles[engine.player.tracking])
            self.level.track(engine.player.mobile)
                        
        for mob in self.level.mobiles.values():
            mob.save()
        
        self.lastHP = engine.player.mobile.health
        
        self.ui.bottomPane.floor = engine.player.floor
        
        self.playersTurn = True
        self.moveElapsed = 0
        self.mode = Mode.DEFAULT
         
        engine.player.save()
        
        self.save.saveFloor(self.level)
        self.save.savePlayer(engine.player)
        self.save.commit()
        
    def loadGame(self):
        self.save = save.Save(self.saveFile)
        self.save.open()
        
        playerData = self.save.getPlayerData()
        self.startGame(playerData)
                
    def bindKey(self, event, cfg, func):
        if cfg is None:
            print "No binding found for function %s" % func.__name__
            return
        
        if cfg.value != None:
            self.input.attachHandler(event, cfg.value, func, cfg.name)
        if cfg.alt != None:
            self.input.attachHandler(event, cfg.alt, func, cfg.name)
            
        self.keyBinds.append(cfg.name)
        
    def unbind(self, event, key, info):
        if self.mode != Mode.OPTIONS:
            return
        
        self.ui.options.unbind()
        
    def readConfig(self):        
        self.volume = (self.config.getValue("Engine", "MasterVolume"), \
                       self.config.getValue("Engine", "SoundVolume"), \
                       self.config.getValue("Engine", "MusicVolume"))
        self.attackOnBump = self.config.getValue("Engine", "AttackOnBump")
        self.attackConfirms = self.config.getValue("Engine", "AttackConfirms")
        self.rememberTarget = self.config.getValue("Engine", "RememberTarget")
        self.multiKeyDiagonals = self.config.getValue("Engine", "MultiKeyDiagonals")
        
        self.bindKeys()
        
    def test(self, event, key, info):
        engine.player.addFormExp(5)
        pass
        
    def bindKeys(self):
        self.input.clearHandlers()
        self.keyBinds = []
        
        #self.input.attachHandler(pygame.KEYDOWN, pygame.K_F6, self.test, info=None)
        
        # special case for options menu unbinding keys
        self.input.attachHandler(pygame.KEYDOWN, pygame.K_BACKSPACE, self.unbind, "Unbind")
        self.input.attachHandler(pygame.KEYDOWN, pygame.K_DELETE, self.unbind, "Unbind")
        
        if self.multiKeyDiagonals:
            self.bindKey(pygame.KEYDOWN, self.config.getOption("KeyBinds", "Up"), self.move)
            self.bindKey(pygame.KEYDOWN, self.config.getOption("KeyBinds", "Down"), self.move)
            self.bindKey(pygame.KEYDOWN, self.config.getOption("KeyBinds", "Left"), self.move)
            self.bindKey(pygame.KEYDOWN, self.config.getOption("KeyBinds", "Right"), self.move)
            self.bindKey(pygame.KEYUP, self.config.getOption("KeyBinds", "Up"), self.movePlayerEnd)
            self.bindKey(pygame.KEYUP, self.config.getOption("KeyBinds", "Down"), self.movePlayerEnd)
            self.bindKey(pygame.KEYUP, self.config.getOption("KeyBinds", "Left"), self.movePlayerEnd)
            self.bindKey(pygame.KEYUP, self.config.getOption("KeyBinds", "Right"), self.movePlayerEnd)
        else:
            self.bindKey(pygame.KEYDOWN, self.config.getOption("KeyBinds", "Up"), self.move)
            self.bindKey(pygame.KEYDOWN, self.config.getOption("KeyBinds", "Down"), self.move)
            self.bindKey(pygame.KEYDOWN, self.config.getOption("KeyBinds", "Left"), self.move)
            self.bindKey(pygame.KEYDOWN, self.config.getOption("KeyBinds", "Right"), self.move)
            
        self.bindKey(pygame.KEYDOWN, self.config.getOption("KeyBinds", "UpLeft"), self.move)
        self.bindKey(pygame.KEYDOWN, self.config.getOption("KeyBinds", "UpRight"), self.move)
        self.bindKey(pygame.KEYDOWN, self.config.getOption("KeyBinds", "DownLeft"), self.move)
        self.bindKey(pygame.KEYDOWN, self.config.getOption("KeyBinds", "DownRight"), self.move)
        
        self.bindKey(pygame.KEYDOWN, self.config.getOption("KeyBinds", "Wait"), self.wait)
        self.bindKey(pygame.KEYDOWN, self.config.getOption("KeyBinds", "WaitOrConfirm"), self.waitOrConfirm)
        
        self.bindKey(pygame.KEYDOWN, self.config.getOption("KeyBinds", "Attack"), self.beginAttack)
        self.bindKey(pygame.KEYDOWN, self.config.getOption("KeyBinds", "Skills"), self.openSkillPane)
        self.bindKey(pygame.KEYDOWN, self.config.getOption("KeyBinds", "Status"), self.openStatsPane)
        self.bindKey(pygame.KEYDOWN, self.config.getOption("KeyBinds", "Form"), self.openFormPane)
        self.bindKey(pygame.KEYDOWN, self.config.getOption("KeyBinds", "Log"), self.showLog)
        
        self.bindKey(pygame.KEYDOWN, self.config.getOption("KeyBinds", "Confirm"), self.confirm)
        self.bindKey(pygame.KEYDOWN, self.config.getOption("KeyBinds", "Cancel"), self.closeUI)

        self.bindKey(pygame.KEYDOWN, self.config.getOption("KeyBinds", "WindowSizeUp"), self.resizeWindow)
        self.bindKey(pygame.KEYDOWN, self.config.getOption("KeyBinds", "WindowSizeDown"), self.resizeWindow)
        
        self.ui.options.keyBinds = self.keyBinds
        
    def getLastTarget(self):
        if self.lastTarget == None:
            return None
        
        o = self.lastTarget()
        
        if o != None and o.dead:
            o = None
        
        return o
    
    def setLastTarget(self, target):
        if target != None:
            self.lastTarget = weakref.ref(target)
        
    def anyKey(self, key):
        if self.mode == Mode.TITLESCREEN:
            self.openMenu()
        elif self.mode == Mode.OPTIONS:
            self.ui.options.bindKey(key)
            
    def resizeWindow(self, event, key, info):
        lastScale = engine.scale
        if info == "WindowSizeDown":
            engine.scale -= 1
        if info == "WindowSizeUp":
            engine.scale += 1
            
        if engine.scale < 1:
            engine.scale = 1
        if engine.scale > 4:
            engine.scale = 4
            
        if engine.scale != lastScale:
            self.renderer.resize(engine.scale)
    
    def closeUI(self, event, key, info):
        if self.mode == Mode.SKILLPANE:
            self.ui.skillPane.hide()
            self.mode = Mode.DEFAULT
        elif self.mode == Mode.STATSPANE:
            self.ui.statsPane.hide()
            self.mode = Mode.DEFAULT
        elif self.mode == Mode.FORMPANE:
            self.ui.formPane.hide()
            self.mode = Mode.DEFAULT
        elif self.mode == Mode.TARGET:
            self.cancelCursor()
        elif self.mode == Mode.POSSESSION:
            self.ui.possessPane.hide()
            self.mode = Mode.DEFAULT
            self.ui.possessPane.mob.cleanUp()
        elif self.mode == Mode.OPTIONS:
            self.ui.options.hide()
            if self.ui.mainMenu.fromMode != Mode.TITLESCREEN:
                self.mode = Mode.INGAMEMENU
            else:
                self.mode = Mode.MAINMENU
            self.ui.mainMenu.show()
            self.applyOptions()
        elif self.mode == Mode.MAINMENU:
            self.quitting = True
        elif self.mode == Mode.INGAMEMENU:
            self.ui.mainMenu.hide()
            self.mode = self.ui.mainMenu.fromMode
            if self.mode == Mode.DEAD:
                self.ui.deathScreen.show()
        elif self.mode == Mode.DEFAULT or self.mode == Mode.DEAD:
            self.openMenu()
            
    def applyOptions(self):
        self.setVolume(self.ui.options.getVolume())
        self.ui.options.apply()
        self.config.save()
        self.readConfig()
            
    def showLog(self, event, key, info):
        if self.mode != Mode.DEFAULT or not self.playersTurn:
            return
        
        self.ui.textLog.switchStyle()
        
    def openMenu(self):
        self.ui.mainMenu.fromMode = self.mode
        
        if self.mode == Mode.DEAD:
            self.ui.deathScreen.hide()
        
        if self.mode != Mode.TITLESCREEN:
            self.mode = Mode.INGAMEMENU
            self.ui.mainMenu.inGame = True
        else:
            self.mode = Mode.MAINMENU
            self.ui.mainMenu.inGame = False
        self.ui.mainMenu.show()
        self.ui.titleScreen.hide()
        
    def openOptions(self):
        self.mode = Mode.OPTIONS
        self.ui.options.show()
        self.ui.mainMenu.hide()
        
    def openStatsPane(self, event, key, info):
        if (self.mode != Mode.DEFAULT and self.mode != Mode.STATSPANE and self.mode != Mode.SKILLPANE and self.mode != Mode.FORMPANE) or not self.playersTurn:
            return
        
        if self.mode != Mode.DEFAULT:
            wasMode = self.mode
            self.closeUI(None, None, None)
            if wasMode == Mode.STATSPANE:
                return
        
        self.mode = Mode.STATSPANE
        self.ui.statsPane.show()
        
    def openFormPane(self, event, key, info):
        if (self.mode != Mode.DEFAULT and self.mode != Mode.STATSPANE and self.mode != Mode.SKILLPANE and self.mode != Mode.FORMPANE) or not self.playersTurn:
            return
        
        if self.mode != Mode.DEFAULT:
            wasMode = self.mode
            self.closeUI(None, None, None)
            if wasMode == Mode.FORMPANE:
                return
        
        self.mode = Mode.FORMPANE
        self.ui.formPane.show()
        
    def openSkillPane(self, event, key, info):
        if (self.mode != Mode.DEFAULT and self.mode != Mode.STATSPANE and self.mode != Mode.SKILLPANE and self.mode != Mode.FORMPANE) or not self.playersTurn:
            return
        
        if self.mode != Mode.DEFAULT:
            wasMode = self.mode
            self.closeUI(None, None, None)
            if wasMode == Mode.SKILLPANE:
                return
        
        self.mode = Mode.SKILLPANE
        self.ui.skillPane.show()
   
    def beginAttack(self, event, key, info):
        if not self.playersTurn:
            return
        
        if self.mode == Mode.DEFAULT:
            self.beginTarget(self.attack)
        elif self.mode == Mode.TARGET and self.attackConfirms:
            self.confirmTarget(event, key, info)
        
    def beginTarget(self, func, source=None, skill=None):
        self.mode = Mode.TARGET
        self.cursor.visible = True
        
        lastTarget = self.getLastTarget()
        
        self.cursor.func = func
        if source == None:
            source = engine.player.mobile.getPosition()
        self.cursor.setSource(source)
        
        if lastTarget != None and self.rememberTarget:
            InRange = self.cursor.tryMove(lastTarget.x, lastTarget.y, True)
            if not InRange:
                self.cursor.setSource(source)
        
        self.cursor.skill = skill
        if self.cursor.skill != None:
            self.cursor.range = skill.range
        else:
            self.cursor.range = 1
       
    def attack(self):
        self.cursor.visible = False
        mob = self.level.mobileAt(self.cursor.x, self.cursor.y)
        
        los = self.level.checkLos((engine.player.mobile.x, engine.player.mobile.y), (self.cursor.x, self.cursor.y))
        
        if mob == None or \
        (mob == engine.player.mobile and self.cursor.skill == None) or \
        not los:
            if not los:
                engine.log("You cannot see that.")
            self.mode = Mode.DEFAULT
            return
        
        self.mode = Mode.ATTACK
        self.target = mob
        self.source = engine.player.mobile
        
        sound = content.getSound("swing")
        if self.cursor.skill != None:
            if self.cursor.skill.swingSound != None:
                sound = content.getSound(self.cursor.skill.swingSound)
            else:
                sound = None
        
        self.playSoundAndWait(sound, self.resolvePlayerAttack)
        
    def resolveAttack(self, source, target, skill=None):
        if skill == None:
            dmg = combat.resolvePhysicalAttack(source, target)
            if dmg != None:
                self.playSoundAndWait(content.getSound("bump"), self.endAttack)
            else:
                self.endAttack()
        else:
            res = combat.resolveSkill(skill, source, target)
            if res.sound != None:
                self.playSoundAndWait(content.getSound(res.sound), self.endAttack)
            else:
                self.endAttack()
                
    def resolvePlayerAttack(self):
        if self.cursor.skill != None:
            if self.cursor.skill.canTarget(self.target):
                engine.player.sp -= self.cursor.skill.cost
            else:
                engine.playSound(engine.content.getSound("nope"))
                self.mode = Mode.DEFAULT
                return
        self.resolveAttack(self.source, self.target, self.cursor.skill)
                
    def mobileAttack(self, activity):
        self.activity = activity
        
        sound = "swing"
        if self.activity.skill != None:
            sound = self.activity.skill.swingSound
        
        self.playSoundAndWait(content.getSound(sound), self.resolveMobileAttack)
        
    def resolveMobileAttack(self):
        self.resolveAttack(self.activity.source, self.activity.target, self.activity.skill)
            
    def endAttack(self):
        if self.mode == Mode.ATTACK:
            self.mode = Mode.DEFAULT
            self.endTurn()
        elif self.mode == Mode.AI:
            self.activity = None
            
        self.target = None
        self.source = None
        
    def playSoundAndWait(self, sound, func):
        if sound == None:
            func()
            return
        self.soundElapsed = 0
        self.curSound = sound
        engine.playSound(self.curSound)
        self.soundFunc = func
        
    def confirmTarget(self, event, key, info):
        if self.cursor.visible:
            self.cursor.func()
        
        if self.rememberTarget:
            self.setLastTarget(self.level.mobileAt(self.cursor.x, self.cursor.y))
            
    def confirmSkillUse(self):
        skill = self.ui.skillPane.getSelectedSkill()
        if engine.player.sp < skill.cost or skill.passive:
            engine.playSound(content.getSound("nope"))
            return
        self.closeUI(None, None, None)
        self.beginTarget(self.useSkill, skill=skill)
        
    def useSkill(self):
        self.cursor.visible = False
        mob = self.level.mobileAt(self.cursor.x, self.cursor.y)
        self.attack()
        
    def confirmStatPoint(self):
        if engine.player.statPoints <= 0:
            return
        
        pos = self.ui.statsPane.cursorPos
        
        if pos == 0:
            if engine.player.health >= engine.player.mobile.formHealth:
                engine.playSound(engine.content.getSound("nope"))
                return
            
            engine.player.health += 3
            
            prevHp = engine.player.mobile.maxHealth
            engine.player.mobile.maxHealth = min(engine.player.mobile.formHealth, engine.player.health)
            dif = prevHp - engine.player.mobile.maxHealth
            engine.player.mobile.health += dif
        elif pos == 1:
            if engine.player.str >= engine.player.mobile.formStr:
                engine.playSound(engine.content.getSound("nope"))
                return
            
            engine.player.str += 1
            engine.player.mobile.str = min(engine.player.mobile.formStr, engine.player.str)
        elif pos == 2:
            if engine.player.agi >= engine.player.mobile.formAgi:
                engine.playSound(engine.content.getSound("nope"))
                return
            
            engine.player.agi += 1
            engine.player.mobile.agi = min(engine.player.mobile.formAgi, engine.player.agi)
        elif pos == 3:
            if engine.player.con >= engine.player.mobile.formCon:
                engine.playSound(engine.content.getSound("nope"))
                return
            
            engine.player.con += 1
            engine.player.mobile.con = min(engine.player.mobile.formCon, engine.player.con)
        elif pos == 4:
            if engine.player.mag >= engine.player.mobile.formMag:
                engine.playSound(engine.content.getSound("nope"))
                return
            
            engine.player.mag += 1
            engine.player.mobile.mag = min(engine.player.mobile.formMag, engine.player.mag)
        elif pos == 5:
            if engine.player.wil >= engine.player.mobile.formWil:
                engine.playSound(engine.content.getSound("nope"))
                return
            
            engine.player.wil += 1
            engine.player.mobile.wil = min(engine.player.mobile.formWil, engine.player.wil)
            
        engine.player.statPoints -= 1
        engine.playSound(content.getSound("uiping"))
        
        engine.player.checkMastery()
        
    def confirmFormPoint(self):
        data = engine.player.getFormData()
        if data.statPoints <= 0:
            return
        
        pos = self.ui.formPane.cursorPos
        
        if pos == 0:
            data.health += 3
            engine.player.mobile.health += 3
            engine.player.mobile.maxHealth += 3
        elif pos == 1:
            data.str += 1
            engine.player.mobile.str += 1
        elif pos == 2:
            data.agi += 1
            engine.player.mobile.agi += 1
        elif pos == 3:
            data.con += 1
            engine.player.mobile.con += 1
        elif pos == 4:
            data.mag += 1
            engine.player.mobile.mag += 1
        elif pos == 5:
            data.wil += 1
            engine.player.mobile.wil += 1
            
        data.statPoints -= 1
        engine.playSound(content.getSound("uiping"))
        
    def confirm(self, event, key, info):
        if self.mode == Mode.TARGET:
            self.confirmTarget(event, key, info)
        elif self.mode == Mode.SKILLPANE:
            self.confirmSkillUse()
        elif self.mode == Mode.STATSPANE:
            self.confirmStatPoint()
        elif self.mode == Mode.FORMPANE:
            self.confirmFormPoint()
        elif self.mode == Mode.POSSESSION:
            self.possess(self.ui.possessPane.mob)
        elif self.mode == Mode.MAINMENU or self.mode == Mode.INGAMEMENU:
            self.confirmMenu()
        elif self.mode == Mode.OPTIONS:
            self.ui.options.confirm()
            if self.ui.options.binding:
                self.input.pressAnyKey(self.anyKey)
            
    def confirmMenu(self):
        if self.ui.mainMenu.cursorPos == 0:  # New Game
            if self.save != None and not self.save.closed:
                self.save.close()
                self.save = None
            self.startGame()
        elif self.ui.mainMenu.cursorPos == 1:  # Continue
            if self.mode == Mode.MAINMENU:
                if os.path.exists(self.saveFile):
                    self.loadGame()
                else:
                    engine.playSound(engine.content.getSound("nope"))
            else:
                if self.ui.mainMenu.fromMode != Mode.DEAD:
                    self.closeUI(None, None, None)
                else:
                    engine.playSound(engine.content.getSound("nope"))
        elif self.ui.mainMenu.cursorPos == 2:  # Options
            self.openOptions()
        elif self.ui.mainMenu.cursorPos == 3:  # Quit
            self.quitting = True
        
    def waitOrConfirm(self, event, key, info):
        if self.mode == Mode.DEFAULT:
            self.wait(event, key, info)
        else:
            self.confirm(event, key, info)
    
    def wait(self, event, key, info):
        if self.mode != Mode.DEFAULT:
            return
        self.endTurn()
        
    def cancelCursor(self):
        self.cursor.visible = False
        self.mode = Mode.DEFAULT
    
    def moveCursor(self, event, key, info):
        if info == "Down":
            self.cursor.tryMove(0, 1)
        elif info == "Up":
            self.cursor.tryMove(0, -1)
        elif info == "Right":
            self.cursor.tryMove(1, 0)
        elif info == "Left":
            self.cursor.tryMove(-1, 0)
        elif info == "UpLeft":
            self.cursor.tryMove(-1, -1)
        elif info == "UpRight":
            self.cursor.tryMove(1, -1)
        elif info == "DownLeft":
            self.cursor.tryMove(-1, 1)
        elif info == "DownRight":
            self.cursor.tryMove(1, 1)
            
    def movePlayerTry(self, event, key, info):
        if info == "UpRight" or info == "UpLeft" or info == "DownRight" or info == "DownLeft":
            return
        
        if self.baseDir == None:
            self.baseDir = info
            self.hasMovedDiagonally = False
            return
        
        if self.baseDir == "Left" and info == "Right" or \
        self.baseDir == "Right" and info == "Left" or \
        self.baseDir == "Up" and info == "Down" or \
        self.baseDir == "Down" and info == "Up":
            return
        
        combined = ""
        if self.baseDir == "Left" and info == "Up":
            combined = "UpLeft"; self.hasMovedDiagonally = True
        elif self.baseDir == "Left" and info == "Down":
            combined = "DownLeft"; self.hasMovedDiagonally = True
        if self.baseDir == "Up" and info == "Left":
            combined = "UpLeft"; self.hasMovedDiagonally = True
        elif self.baseDir == "Up" and info == "Right":
            combined = "UpRight"; self.hasMovedDiagonally = True
        if self.baseDir == "Right" and info == "Up":
            combined = "UpRight"; self.hasMovedDiagonally = True
        elif self.baseDir == "Right" and info == "Down":
            combined = "DownRight"; self.hasMovedDiagonally = True
        if self.baseDir == "Down" and info == "Left":
            combined = "DownLeft"; self.hasMovedDiagonally = True
        elif self.baseDir == "Down" and info == "Right":
            combined = "DownRight"; self.hasMovedDiagonally = True
        
        if combined != "":
            self.movePlayer(event, key, combined)
            
    def movePlayerEnd(self, event, key, info):
        if info == self.baseDir:
            if not self.hasMovedDiagonally:
                self.movePlayer(event, key, info)
            
            self.baseDir = None
        
    def movePlayer(self, event, key, info):
        res = True
        if info == "Down":
            res = self.level.movePlayer(Direction.DOWN)
        elif info == "Up":
            res = self.level.movePlayer(Direction.UP)
        elif info == "Right":
            res = self.level.movePlayer(Direction.RIGHT)
        elif info == "Left":
            res = self.level.movePlayer(Direction.LEFT)
        elif info == "UpLeft":
            res = self.level.movePlayer(Direction.UPLEFT)
        elif info == "UpRight":
            res = self.level.movePlayer(Direction.UPRIGHT)
        elif info == "DownLeft":
            res = self.level.movePlayer(Direction.DOWNLEFT)
        elif info == "DownRight":
            res = self.level.movePlayer(Direction.DOWNRIGHT)
            
        if res != None:
            if isinstance(res, mobs.Mobile) and self.attackOnBump:
                self.cursor.x = res.x
                self.cursor.y = res.y
                self.cursor.skill = None
                self.setLastTarget(res)
                self.attack()
            else:
                engine.playSound(content.getSound("bump"))
        else:
            tile = self.level.getTile(engine.player.mobile.x, engine.player.mobile.y)
            if tile.type == TileType.STAIRSDOWN or tile.type == TileType.STAIRSUP:
                self.climbStairs(tile.type)
            else:
                self.endTurn()
                            
    def move(self, event, key, info):
        if self.mode == Mode.MAINMENU or self.mode == Mode.INGAMEMENU:
            self.ui.mainMenu.moveCursor(event, key, info)
            return
        elif self.mode == Mode.OPTIONS:
            self.ui.options.moveCursor(event, key, info)
            return

        if not self.playersTurn:
            return
        
        if self.mode == Mode.TARGET:
            self.moveCursor(event, key, info)
            return
        elif self.mode == Mode.SKILLPANE:
            self.ui.skillPane.moveCursor(event, key, info)
            return
        elif self.mode == Mode.STATSPANE:
            if engine.player.statPoints > 0:
                self.ui.statsPane.moveCursor(event, key, info)
            return
        elif self.mode == Mode.FORMPANE:
            if engine.player.getFormPoints() > 0:
                self.ui.formPane.moveCursor(event, key, info)
            return
        elif self.mode == Mode.POSSESSION:
            self.ui.possessPane.moveCursor(event, key, info)
            return
        elif self.mode != Mode.DEFAULT:
            return
        
        if self.multiKeyDiagonals:
            self.movePlayerTry(event, key, info)
        else:
            self.movePlayer(event, key, info)
                
    def climbStairs(self, type):
        if type == TileType.STAIRSDOWN:
            engine.player.floor += 1
            engine.player.exploredFloors = max(engine.player.floor, engine.player.exploredFloors)
        else:
            engine.player.floor -= 1
            
        if len(self.levels) < engine.player.floor:
            newLevel = maps.generate(engine.roomSize, engine.floorWidth, engine.floorHeight, self.level.maxHazard, False, engine.player.floor)
            self.levels.append(newLevel)
            self.save.addFloor(newLevel)
            
        self.renderer.renderables.remove(self.level.renderer)
        engine.player.mobile.cleanUp()
        
        self.level = self.levels[engine.player.floor - 1]
        self.level.addMobile(engine.player.mobile)
        self.level.track(engine.player.mobile)
        self.mobiles = list(self.level.mobiles.values())
        
        self.level.setDeathCallback(self.mobileDied)
        self.renderer.renderables.insert(0, self.level.renderer)
        
        if type == TileType.STAIRSDOWN:
            engine.player.mobile.moveTo(self.level.stairsUp[0], self.level.stairsUp[1])
        else:
            engine.player.mobile.moveTo(self.level.stairsDown[0], self.level.stairsDown[1])
            
        self.ui.bottomPane.floor = engine.player.floor
        
    def endTurn(self):
        self.playersTurn = False
        self.mobiles = list(self.level.mobiles.values())
        self.moveElapsed = 0
        self.mode = Mode.ENDTURN
        
    def beginTurn(self):
        if engine.player.mobile.dead:
            return
        
        self.playersTurn = True
        engine.player.regenSp()
        self.moveElapsed = 0
        self.mode = Mode.DEFAULT
        
        for mobile in self.level.mobiles.values():
            mobile.updateStatus()
        
        if engine.player.mobile.hasCondition(Condition.POISON):
            engine.playSound(content.getSound("envenom"))
            engine.log("Poisoned!")
    
        engine.player.save()
        
        if self.save != None and not self.save.closed:
            self.save.saveFloor(self.level)
            self.save.savePlayer(engine.player)
            self.save.commit()
        
    def mobileDied(self, mob, source):
        if mob == engine.player.mobile:
            engine.log("Your form has dissipated")
            
            if engine.player.mobile.getName() == "Imp":
                self.mode = Mode.DEAD
                self.ui.deathScreen.show()
                self.save.close()
                if os.path.exists(self.saveFile):
                    os.remove(self.saveFile)
                return

            imp = mobs.Imp(engine.player.mobile.x, engine.player.mobile.y)
            
            if mob.health > 0:
                perc = 1.0 * mob.health / mob.maxHealth
            else:
                perc = 0.05
                
            engine.player.possess(imp)
            self.level.addMobile(imp)
            self.level.track(imp)
            self.setPossessStats(imp)
            
            imp.health = max(int(round(perc * imp.maxHealth)), 1)
        elif source == engine.player.mobile:
            engine.log("%s has died!" % mob.getName())
            
            engine.player.gainExpFromMobile(mob)
            
            if mob.hasCondition(Condition.VESSEL):
                self.beginPossess(mob)

    def beginPossess(self, target):
        self.ui.possessPane.choice = True
        self.mode = Mode.POSSESSION
        self.ui.possessPane.show()
        self.ui.possessPane.mob = target
        target.dead = False
        
    def setPossessStats(self, target):
        target.maxHealth = min(engine.player.health, target.formHealth)
        target.str = min(engine.player.str, target.formStr)
        target.agi = min(engine.player.agi, target.formAgi)
        target.con = min(engine.player.con, target.formCon)
        target.mag = min(engine.player.mag, target.formMag)
        target.wil = min(engine.player.wil, target.formWil)
        
        target.maxHealth += engine.player.masterHealth
        target.str += engine.player.masterStr
        target.agi += engine.player.masterAgi
        target.con += engine.player.masterCon
        target.mag += engine.player.masterMag
        target.wil += engine.player.masterWil
        
        data = engine.player.getFormData(target.getName())
        target.maxHealth += data.health
        target.str += data.str
        target.agi += data.agi
        target.con += data.con
        target.mag += data.mag
        target.wil += data.wil
        
    def possess(self, target):
        if self.ui.possessPane.choice:
            engine.log("You have possessed %s." % target.getName())
            engine.player.mobile.cleanUp()
            
            self.setPossessStats(target)
        
            target.health = target.maxHealth
            
            target.ai = None
            engine.player.possess(target)
            self.level.track(engine.player.mobile)
        else:
            target.cleanUp()
        
        # grant remaining level when entering new unmastered form
        # currently makes the start too hard
        #engine.player.gainExp(0)
            
        self.ui.possessPane.hide()
        self.mode = Mode.DEFAULT
        self.endTurn()
    
    def run(self):
        while True:
            events = pygame.event.get()
            
            for event in events:
                if event.type == pygame.QUIT:
                    self.quitting = True
                    break
                
                if event.type == pygame.KEYDOWN or event.type == pygame.KEYUP:
                    self.input.tryHandle(event.type, event.key)
                    
            if self.quitting:
                break
            
            curTick = pygame.time.get_ticks()
            elapsed = curTick - self.lastTick;
            self.lastTick = curTick
            
            self.update(elapsed)
            self.beginDraw(elapsed)
            
        if self.save != None:
            self.save.close()
        
        self.config.save()
            
    def waitOnSound(self, sound, elapsed):
        duration = sound.get_length() * 1000
        self.soundElapsed += elapsed
        if self.soundElapsed > duration:
            self.curSound = None
            self.soundFunc()
            
    def setVolume(self, volumes):
        master = volumes[0]
        sfx = volumes[1]
        music = volumes[2]
        
        self.volume = volumes
        sound.masterVolume = master / 100.0
        sound.soundVolume = sfx / 100.0
        musicVol = music / 100.0
        
        if self.music != None:
            self.music.setMaxVolume(musicVol)
            self.music.setMasterVolume(sound.masterVolume)
        pygame.mixer.music.set_volume(sound.masterVolume * musicVol)
        for i in xrange(2, pygame.mixer.get_num_channels()):
            channel = pygame.mixer.Channel(i)
            channel.set_volume(sound.masterVolume * sound.soundVolume)
        
    def update(self, elapsed):
        if self.mode == Mode.OPTIONS:
            self.setVolume(self.ui.options.getVolume())
        
        if self.music != None:
            if engine.player.mobile != None and \
            (engine.player.mobile.health < engine.player.mobile.maxHealth / 3 or \
             engine.player.mobile.health < self.lastHP):
                self.lastHP = engine.player.mobile.health
                self.music.addEvent(music.EventDamage())
            self.music.update(elapsed)
                
        if self.curSound != None:
            self.waitOnSound(self.curSound, elapsed)
            return
        
        if not self.playersTurn and not self.mode == Mode.ENDTURN:
            self.moveElapsed += elapsed
            
            activity = None
            while self.moveElapsed >= self.moveSpeed:
                self.moveElapsed -= self.moveSpeed
                
                while activity == None:
                    mob = self.popMobile()
                    if mob == None:
                        break
                    
                    if not mob.dead:
                        dist = self.level.distanceToPlayer(mob)
                        if dist < engine.roomSize * 5 / 2:
                            if dist > engine.roomSize:
                                mob.update()
                            else:
                                activity = mob.update()
                        
                if activity != None and activity.type == ActivityType.ATTACK:
                    self.mobileAttack(activity)
                    return
                
            if len(self.mobiles) == 0:
                self.beginTurn()
        
        # input happens before update() is called
        # this means that the player moves and ends their turn, and immediately thereafter
        # the AI starts moving, but line of sight is set in render, which comes after.
        # so theoretically an AI could think it doesn't have line of sight, even though it
        # would get it a frame later. This delays the AI by that one frame.
        if self.mode == Mode.ENDTURN:
            self.mode = Mode.AI
        
    def popMobile(self):
        if len(self.mobiles) > 0:
            return self.mobiles.pop(0)
        else:
            return None
    
    def beginDraw(self, elapsed):
        self.renderer.render()
        
    def render(self, screen):
        if self.cursor.visible:
            drawPos = self.level.renderer.camera.translate(self.cursor.x, self.cursor.y)
            screen.blit(content.getImage("reticule"), drawPos)

game = Game()
engine.setGameRef(game)
game.run()
