import random
import weakref

from enum import *

import pathing
import engine
import mobs
import mathhelper

class Activity(object):
    def __init__(self, type, target=None):
        self.type = type
        self.target = target
        
class ActMove(Activity):
    def __init__(self):
        Activity.__init__(self, ActivityType.MOVE)

class ActAttack(Activity):
    def __init__(self, source, target, skill=None):
        Activity.__init__(self, ActivityType.ATTACK)
        self.source = source
        self.target = target
        self.skill = skill

class AI(object):
    def __init__(self, parent):
        self.parent = weakref.ref(parent)
        self.path = None
        self.aggro = False
        self.blockedPos = []
        self.skillChance = 0.2
        self.moveChance = 0.2
        self.maxMoveLength = 5
    
    def tick(self):
        for pos in self.blockedPos:
            if self.parent().world().checkCanMove(pos[0], pos[1]) == None:
                self.blockedPos.remove(pos)
        
        if self.path != None and len(self.path) > 0:
            coords = self.path.pop(0)
            if self.parent().moveTo(coords[0], coords[1]) != None:
                self.path.insert(0, coords)
                if not coords in self.blockedPos:
                    self.blockedPos.append(coords)
            else:
                if len(self.path) == 0:
                    self.path = None
                return ActMove()
        
        return None
    
    def pathTo(self, x, y):
        aStar = pathing.AStar((self.parent().x, self.parent().y),
                              (x, y), self.parent().world().getPathingCell,
                              blockedCells=self.blockedPos)
        self.path = aStar.process()
        
    def attack(self, mobile):
        attack = ActAttack(self.parent(), mobile)
        
        eligibleSkills = self.parent().skills[:]
        
        for i in reversed(xrange(len(eligibleSkills))):
            if eligibleSkills[i].passive:
                del eligibleSkills[i]
        
        if len(eligibleSkills) > 0:
            if mathhelper.success(self.skillChance):
                attack.skill = eligibleSkills[random.randint(0, len(eligibleSkills) - 1)]
                
        return attack
    
    def inRange(self, maxDist, roughDist):
        # TODO: With los this function might be pointless?
        if self.parent().hasLos and maxDist <= roughDist * 2:
            return self.parent().world().inRangeOfPlayer(self.parent(), maxDist)
        return False
    
    def meander(self):
        if self.path == None:
            if random.random() < self.moveChance:
                steps = random.randint(1, self.maxMoveLength)
                direction = random.randint(0, 7)
                self.path = []
                
                curX = self.parent().x
                curY = self.parent().y
                
                for step in xrange(steps):
                    if direction == Direction.DOWN:
                        curY += 1
                    elif direction == Direction.UP:
                        curY -= 1
                    elif direction == Direction.RIGHT:
                        curX += 1
                    elif direction == Direction.LEFT:
                        curX -= 1
                    elif direction == Direction.UPLEFT:
                        curX -= 1
                        curY -= 1
                    elif direction == Direction.UPRIGHT:
                        curX += 1
                        curY -= 1
                    elif direction == Direction.DOWNLEFT:
                        curX -= 1
                        curY += 1
                    elif direction == Direction.DOWNRIGHT:
                        curX += 1
                        curY += 1
                    
                    if self.parent().world().checkCanMove(curX, curY) == None and \
                        self.parent().checkCanMove(self.parent().world().getTile(curX, curY)) == None:
                            self.path.append((curX, curY))
                    else:
                        break
        
class Aggressive(AI):
    def __init__(self, parent, scanDist=7, detectionChance=0.25):
        AI.__init__(self, parent)
        self.scanDist = scanDist
        self.detectionChance = detectionChance
        
    def tick(self):
        dist = self.scanForPlayer()
        if dist > -1:
            if self.detectPlayer():
                if not self.aggro:
                    self.parent().aggro()
                if self.inRange(1, dist) and self.parent().world().checkLos((self.parent().x, self.parent().y), (engine.player.mobile.x, engine.player.mobile.y)):
                    return self.attack(engine.player.mobile)
                else:
                    self.pathTo(engine.player.mobile.x, engine.player.mobile.y)
        else:
            self.aggro = False
            self.meander()
        
        return AI.tick(self)
    
    def detectPlayer(self):
        if self.aggro or random.random() < self.detectionChance:
            return True
        return False
        
    def scanForPlayer(self):
        # TODO: With los this function might be pointless?
        if not self.parent().hasLos:
            return -1
        scanDist = self.scanDist
        if self.aggro:
            scanDist *= 2
        dist = self.parent().world().distanceToPlayer(self.parent())
        if dist <= scanDist:
            return dist
        return -1
    
class Mage(Aggressive):
    def __init__(self, parent, scanDist=7, detectionChance=0.5):
        Aggressive.__init__(self, parent, scanDist)
        self.skillChance = 0.66

    def tick(self):
        dist = self.scanForPlayer()
        if dist > -1:
            if self.detectPlayer():
                if not self.aggro:
                    self.parent().aggro()
                if self.inRange(1, dist) and self.parent().world().checkLos((self.parent().x, self.parent().y), (engine.player.mobile.x, engine.player.mobile.y)):
                    return self.attack(engine.player.mobile)
                else:
                    if self.parent().world().checkLos((self.parent().x, self.parent().y), (engine.player.mobile.x, engine.player.mobile.y)):
                        eligibleSkills = self.parent().skills[:]
                                
                        for i in reversed(xrange(len(eligibleSkills))):
                            if eligibleSkills[i].passive or not self.inRange(eligibleSkills[i].range, dist):
                                del eligibleSkills[i]
                                
                        if len(eligibleSkills) > 0:
                            if mathhelper.success(self.skillChance):
                                attack = ActAttack(self.parent(), engine.player.mobile)
                                attack.skill = eligibleSkills[random.randint(0, len(eligibleSkills) - 1)]
                                return attack
                        
                    self.pathTo(engine.player.mobile.x, engine.player.mobile.y)
        else:
            self.aggro = False
            self.meander()
                    
        return AI.tick(self)

class Passive(AI):
    def __init__(self, parent):
        AI.__init__(self, parent)
        
    def tick(self):
        self.meander()
        
        if self.path != None and len(self.path) == 0:
            self.path = None
           
        return AI.tick(self)