import random
import math
import sys
import pygame

from enum import *

import render
import mobs
import spawning
import engine
import pathing
import mathhelper

roomSize = 0

class Tile(object):
	def __init__(self):
		self.type = TileType.NONE
		self.style = 0
		self.visited = False
		
class TileOutOfBounds(Tile):
	def __init__(self):
		Tile.__init__(self)
		self.type = TileType.OUTOFBOUNDS
		
class FloorTile(Tile):
	def __init__(self):
		Tile.__init__(self)
		self.type = TileType.FLOOR
		
class WallTile(Tile):
	def __init__(self):
		Tile.__init__(self)
		self.type = TileType.WALL

class StairsUp(Tile):
	def __init__(self):
		Tile.__init__(self)
		self.type = TileType.STAIRSUP
		
class StairsDown(Tile):
	def __init__(self):
		Tile.__init__(self)
		self.type = TileType.STAIRSDOWN
		
class Room(object):
	def __init__(self, size, x, y):
		self.connected = False
		self.connections = []
		self.x = x
		self.y = y
		self.neighbors = None
		self.size = size
		self.tiles = [[None]*self.size for i in xrange(self.size)]
		self.center = (0, 0)
		self.hazard = 0
		self.floor = None # for spawning
		
	def addConnection(self, room):
		if not (room.x, room.y) in self.connections:
			self.connections.append((room.x, room.y))

	def setTile(self, x, y, tile):
		self.tiles[y][x] = tile
		
	def getTile(self, x, y):
		if self.tiles[y][x] == None:
			self.tiles[y][x] = Tile()
		return self.tiles[y][x]
			
class Level(object):
	def __init__(self, cellSize, width, height, baseHazard, number, seed):
		self.cellSize = cellSize
		self.cellWidth = width
		self.cellHeight = height
		self.tileWidth = width * cellSize
		self.tileHeight = height * cellSize
		self.rooms = None
		self.renderer = render.LevelRenderer(self)
		self.mobiles = {}
		self.onMobileDeath = None
		self.baseHazard = baseHazard
		self.stairsDown = (-1, -1)
		self.stairsUp = (-1, -1)
		self.floorNumber = 1
		self.seed = seed
		self.number = number
		self.tileMobiles = {}
		self.losMobs = []
		self.maxHazard = 0
		
		self.saveData = { "version": 0,
						"seed": self.seed,
						 "cellSize": self.cellSize,
						 "cellWidth": self.cellWidth,
						 "cellHeight": self.cellHeight, 
						 "baseHazard": self.baseHazard,
						 "number": self.number,
						 "seed": self.seed,
						 "mobiles": {} }
		
	def cleanUp(self):
		mobiles = self.mobiles.values()
		for i in reversed(xrange(len(mobiles))):
			mobiles[i].cleanUp()
		
		self.tileMobiles.clear()
		self.renderer.cleanUp()
		
		del self.rooms[:]
		del self.renderer
		self.mobiles.clear()
		self.saveData.clear()
				
	def setStairs(self, x, y, tile, down):
		if down:
			self.stairsDown = (x, y)
		else:
			self.stairsUp = (x, y)
			
		self.setTile(x, y, tile)
		
	def setDeathCallback(self, func):
		self.onMobileDeath = func
		
	def setRooms(self, rooms):
		self.rooms = rooms
		
	def cellCoords(self, x, y):
		return (int(x / self.cellSize), int(y / self.cellSize), x % self.cellSize, y % self.cellSize)
		
	def getTile(self, x, y):
		if x < 0 or y < 0 or x >= self.tileWidth or y >= self.tileHeight:
			return TileOutOfBounds()
		coords = self.cellCoords(x, y)
		return self.rooms[coords[1]][coords[0]].getTile(coords[2], coords[3])
	
	def getPathingCell(self, x, y, homeCell=False):
		tile = self.getTile(x, y)
		if tile.type == TileType.OUTOFBOUNDS:
			return None
		return pathing.Cell(x, y, tile.type == TileType.FLOOR)
	
	def setTile(self, x, y, tile):
		coords = self.cellCoords(x, y)
		self.rooms[coords[1]][coords[0]].setTile(coords[2], coords[3], tile)
		
	def getTileArea(self, x, y):
		tiles = [[None]*3 for i in xrange(3)]
		tiles[0][0] = self.getTile(x - 1, y - 1)
		tiles[1][0] = self.getTile(x, y - 1)
		tiles[2][0] = self.getTile(x + 1, y - 1)
		tiles[0][1] = self.getTile(x - 1, y)
		tiles[1][1] = self.getTile(x, y)
		tiles[2][1] = self.getTile(x + 1, y)
		tiles[0][2] = self.getTile(x - 1, y + 1)
		tiles[1][2] = self.getTile(x, y + 1)
		tiles[2][2] = self.getTile(x + 1, y + 1)
		return tiles
	
	def addMobile(self, mob):
		self.mobiles[mob.ID] = mob
		self.saveData["mobiles"][mob.ID] = mob.saveData
		mob.setWorld(self)
		self.addMobileToTile(mob)
		
	def addMobileToTile(self, mob, p=None):
		if p == None:
			p = (mob.x, mob.y)
		if not p in self.tileMobiles:
			self.tileMobiles[p] = [mob]
		else:
			self.tileMobiles[p].append(mob)
			
	def removeMobileFromTile(self, mob, p=None):
		if p == None:
			p = (mob.x, mob.y)
		self.tileMobiles[p].remove(mob)
		
	def getMobileByID(self, ID):
		if ID in self.mobiles:
			return self.mobiles[ID]
		return None
	
	def mobileDied(self, mob, source):
		self.onMobileDeath(mob, source)
		
	def mobileMoved(self, mob, fromPos, toPos):
		self.removeMobileFromTile(mob, fromPos)
		self.addMobileToTile(mob, toPos)
	
	def removeMobile(self, mob):
		self.removeMobileFromTile(mob)
		del self.mobiles[mob.ID]
		del self.saveData["mobiles"][mob.ID]
		
	def track(self, mob):
		self.renderer.camera.target = mob
		
	def movePlayer(self, direction):
		return engine.player.mobile.move(direction)
	
	def moveMobileTo(self, mobile, x, y):
		mobile.moveTo(x, y)
	
	def distanceToPlayer(self, mob):
		return engine.dist(mob.x, mob.y, engine.player.mobile.x, engine.player.mobile.y)
	
	def inRangeOfPlayer(self, mob, maxDist):
		return abs(mob.x - engine.player.mobile.x) <= maxDist and abs(mob.y - engine.player.mobile.y) <= maxDist
	
	def checkCanMove(self, x, y):
		for mobile in self.mobiles.values():
			if mobile.x == x and mobile.y == y:
				return mobile
		return None
	
	def mobileAt(self, x, y):
		l = self.mobilesAt(x, y)
		if len(l) > 0:
			return l[0]
		else:
			return None
	
	def mobilesAt(self, x, y):
		if (x, y) in self.tileMobiles:
			return self.tileMobiles[(x, y)]
		return []
	
	def checkLos(self, p0, p1):
		ray = mathhelper.Ray(p0, p1)
		ray.cast()
		for point in ray.points:
			tile = self.getTile(point[0], point[1])
			if tile == None or tile.type == TileType.WALL or tile.type == TileType.NONE:
				return False
		return True
	
	def __str__(self):
		str = ""
		for y in xrange(self.tileHeight):
			for x in xrange(self.tileWidth):
				tileType = self.getTile(x, y).type
				
				coords = self.cellCoords(x, y)
				room = self.rooms[coords[1]][coords[0]]
				
				if x - coords[0] * self.cellSize == room.center[0] and y - coords[1] * self.cellSize == room.center[1]:
					str += "O"
				elif tileType == TileType.FLOOR:
					str += "."
				elif tileType == TileType.WALL:
					str += "#"
				else:
					str += " "
			str += "\n"
		return str
		
def selectRoom(rooms, x, y):
	if x < 0 or x >= len(rooms[0]) or y < 0 or y >= len(rooms):
		return None
	if rooms[y][x] == None:
		rooms[y][x] = Room(roomSize, x, y)
	return rooms[y][x]

def getNeighbors(rooms, room):
	if room.neighbors != None:
		return room.neighbors
	
	neighbors = []
	left = selectRoom(rooms, room.x - 1, room.y)
	top = selectRoom(rooms, room.x, room.y - 1)
	right = selectRoom(rooms, room.x + 1, room.y)
	bottom = selectRoom(rooms, room.x, room.y + 1)
	
	if left != None:
		neighbors.append(left)
	if top != None:
		neighbors.append(top)
	if right != None:
		neighbors.append(right)
	if bottom != None:
		neighbors.append(bottom)
		
	room.neighbors = neighbors
	return neighbors

def unconnectedNeighbors(neighbors):
	i = 0
	for room in neighbors:
		if not room.connected:
			i += 1
	return i

def getConnectedNeighbors(rooms, room):
	neighbors = getNeighbors(rooms, room)
	connNeighbors = []
	for neighbor in neighbors:
		if neighbor.connected:
			connNeighbors.append(neighbor)
	return connNeighbors

def connectRooms(fromRoom, toRoom):
	fromRoom.connected = True
	toRoom.connected = True

	fromRoom.addConnection(toRoom)
	toRoom.addConnection(fromRoom)
		
	return toRoom

def fillRoom(room):
	minSize = 3
	
	p0 = (random.randint(0, (roomSize-1-minSize) / 2), random.randint(0, (roomSize-1-minSize) / 2))
	p1 = (p0[0] + random.randint(minSize, roomSize - p0[0]), p0[1] + random.randint(minSize, roomSize - p0[1]))
	
	for y in xrange(p0[1], p1[1]):
		for x in xrange(p0[0], p1[0]):
			room.setTile(x, y, FloorTile())
			
	room.center = ((p0[0] + p1[0]) / 2, (p0[1] + p1[1]) / 2)
	room.floor = pygame.Rect(p0[0], p0[1], p1[0] - p0[0], p1[1] - p0[1]) # for spawning
	
def createCorridors(level):
	for row in level.rooms:
		for room in row:
			neighbors = getConnectedNeighbors(level.rooms, room)
			
			for neighbor in neighbors:
				if neighbor.x >= room.x and neighbor.y >= room.y:
					pStart = (room.x * level.cellSize + room.center[0], room.y * level.cellSize + room.center[1])
					pEnd = (neighbor.x * level.cellSize + neighbor.center[0], neighbor.y * level.cellSize + neighbor.center[1])
					
					xFirst = random.random() > 0.5
					
					if xFirst:
						for x in xrange(min(pStart[0], pEnd[0]), max(pStart[0], pEnd[0]) + 1):
							level.setTile(x, pStart[1], FloorTile())
						
						for y in xrange(min(pStart[1], pEnd[1]), max(pStart[1], pEnd[1]) + 1):
							level.setTile(pEnd[0], y, FloorTile())
					else:
						for y in xrange(min(pStart[1], pEnd[1]), max(pStart[1], pEnd[1]) + 1):
							level.setTile(pStart[0], y, FloorTile())
						
						for x in xrange(min(pStart[0], pEnd[0]), max(pStart[0], pEnd[0]) + 1):
							level.setTile(x, pEnd[1], FloorTile())
						
def createWalls(level):
	for y in xrange(level.tileHeight):
		for x in xrange(level.tileWidth):
			tiles = level.getTileArea(x, y)
			
			if tiles[1][1].type == TileType.NONE:
				if tiles[0][0].type == TileType.FLOOR or \
				tiles[1][0].type == TileType.FLOOR or \
				tiles[2][0].type == TileType.FLOOR or \
				tiles[0][1].type == TileType.FLOOR or \
				tiles[2][1].type == TileType.FLOOR or \
				tiles[0][2].type == TileType.FLOOR or \
				tiles[1][2].type == TileType.FLOOR or \
				tiles[2][2].type == TileType.FLOOR:
					level.setTile(x, y, WallTile())
			elif tiles[1][1].type == TileType.FLOOR and \
			x == 0 or y == 0 or x == level.tileWidth - 1 or y == level.tileHeight - 1:
				level.setTile(x, y, WallTile())
					
	for y in xrange(level.tileHeight):
		for x in xrange(level.tileWidth):
			tiles = level.getTileArea(x, y)
			
			if tiles[1][1].type == TileType.FLOOR:
				tiles[1][1].style = random.randint(0, 3)
			elif tiles[1][1].type == TileType.WALL:
				if tiles[1][2].type == TileType.WALL and tiles[2][1].type == TileType.WALL and \
			 	(tiles[2][2].type != TileType.WALL or \
				not ((tiles[0][1].type == TileType.WALL and tiles[0][2].type == TileType.WALL) or (tiles[1][0].type == TileType.WALL and tiles[2][0].type == TileType.WALL))):
					tiles[1][1].style = 2
				elif tiles[0][1].type == TileType.WALL and tiles[1][0].type == TileType.WALL and \
				tiles[1][2].type != TileType.WALL and tiles[2][1].type != TileType.WALL:
					tiles[1][1].style = 3
					
				elif tiles[2][1].type == TileType.WALL and tiles[0][1].type == TileType.NONE and \
				tiles[1][2].type != TileType.WALL:
					tiles[1][1].style = 4
				elif tiles[2][1].type == TileType.WALL and tiles[0][1].type == TileType.FLOOR and \
				tiles[1][2].type != TileType.WALL:
					tiles[1][1].style = 6
										
				elif tiles[1][2].type == TileType.WALL and tiles[1][0].type == TileType.NONE and \
				tiles[2][1].type != TileType.WALL:
					tiles[1][1].style = 5
				elif tiles[1][2].type == TileType.WALL and tiles[1][0].type == TileType.FLOOR and \
				tiles[2][1].type != TileType.WALL:
					tiles[1][1].style = 7
					
				elif tiles[0][1].type == TileType.WALL and tiles[2][1].type == TileType.NONE and \
				tiles[1][2].type != TileType.WALL:
					tiles[1][1].style = 8
				elif tiles[0][1].type == TileType.WALL and tiles[2][1].type == TileType.FLOOR and \
				tiles[1][2].type != TileType.WALL:
					tiles[1][1].style = 10
										
				elif tiles[1][0].type == TileType.WALL and tiles[1][2].type == TileType.NONE and \
				tiles[2][1].type != TileType.WALL:
					tiles[1][1].style = 8
				elif tiles[1][0].type == TileType.WALL and tiles[1][2].type == TileType.FLOOR and \
				tiles[2][1].type != TileType.WALL:
					tiles[1][1].style = 11
					
				elif tiles[1][0].type != TileType.WALL or tiles[1][2].type != TileType.WALL:
					tiles[1][1].style = 0
				elif tiles[0][1].type != TileType.WALL or tiles[2][1].type != TileType.WALL:
					tiles[1][1].style = 1
					
def setHazard(room, distance, baseHazard):
	room.hazard = distance + random.randint(-1, +1) + baseHazard
	return room.hazard
	
def addStairs(level, room, down):
	if down:
		tile = StairsDown()
	else:
		tile = StairsUp()
	
	level.setStairs(room.x * level.cellSize + room.center[0], room.y * level.cellSize + room.center[1], tile, down)

def generate(cellSize, width, height, baseHazard=0, firstRun=True, floorNumber=1, seed=None, fromSave=False):
	global roomSize
	
	if seed == None:
		random.seed()
		seed = random.randint(0, sys.maxint)
	
	random.seed(seed)
		
	level = Level(cellSize, width, height, baseHazard, floorNumber, seed)
	level.floorNumber = floorNumber
	roomSize = cellSize
	
	rooms = [[None]*width for x in xrange(height)]
	startX = random.randint(0, width-1)
	startY = random.randint(0, height-1)
	
	room = selectRoom(rooms, startX, startY)
	room.connected = True
	neighbors = getNeighbors(rooms, room)
	lastRoom = None
	
	while unconnectedNeighbors(neighbors) > 0:
		room = connectRooms(room, neighbors[random.randint(0, len(neighbors) - 1)])
		lastRoom = room
		neighbors = getNeighbors(rooms, room)
		
	unconnectedRooms = 1
	while unconnectedRooms > 0:
		unconnectedRooms = 0
		for y in xrange(height):
			for x in xrange(width):
				room = selectRoom(rooms, x, y)
				if not room.connected:
					neighbors = getConnectedNeighbors(rooms, room)
					
					#favor connections with least connections
					leastConnectedNeighbors = []
					
					# find minimum number of connections
					minCons = 10
					for neighbor in neighbors:
						minCons = min(minCons, len(neighbor.connections))
						
					for neighbor in neighbors:
						if len(neighbor.connections) == minCons:
							leastConnectedNeighbors.append(neighbor)
					
					if len(leastConnectedNeighbors) > 0:
						neighbor = leastConnectedNeighbors[random.randint(0, len(leastConnectedNeighbors) - 1)]
						connectRooms(room, neighbor)
					else:
						unconnectedRooms += 1
						
					lastRoom = room
	
	maxDist = 0
	for row in rooms:
		for room in row:
			maxDist = max(engine.dist(room.x, room.y, startX, startY), maxDist)
	maxDist = float(maxDist)
		
	maxHazard = 0
	for row in rooms:
		for room in row:
			hazard = int(round(engine.dist(room.x, room.y, startX, startY) / maxDist * 2))
			setHazard(room, hazard, baseHazard)
			maxHazard = max(hazard, maxHazard)
			fillRoom(room)
	level.maxHazard = maxHazard + baseHazard
		
	level.setRooms(rooms)
	createCorridors(level)
	createWalls(level)
		
	startPos = selectRoom(rooms, startX, startY).center
	startPos = (startX * roomSize + startPos[0], startY * roomSize + startPos[1])
	
	if floorNumber > 1:
		addStairs(level, selectRoom(rooms, startX, startY), False)
	addStairs(level, lastRoom, True)
	
	if fromSave:
		return level
	
	for row in level.rooms:
		for room in row:
			spawning.spawn(level, room)
			
	for mob in level.mobiles.values():
		mob.save()
	
	if firstRun:
		player = mobs.Imp(startPos[0], startPos[1])
		level.addMobile(player)
		engine.player.possess(player)
		level.track(player)
		
	if floorNumber == 12:
		room = rooms[random.randint(0, width - 1)][random.randint(0, height - 1)]
		balrog = mobs.spawnByName("Balrog", room.x * level.cellSize + room.center[0], room.y * level.cellSize + room.center[1])
		level.addMobile(balrog)
			
	return level

def load(data):
	level = generate(data["cellSize"], data["cellWidth"], data["cellHeight"], data["baseHazard"], False, data["number"], data["seed"], True)
	
	for mob in data["mobiles"].values():
		level.addMobile(mobs.loadFromData(mob, level))
	
	return level