import weakref
import operator

from enum import *

import engine
import maps
import content
import ai
import skills

curID = 0
    
hazardRatings = { "PlagueRat": 1,
                 "VampireBat": 2,
                 "Skeleton": 3,
                 "Mortipede": 4,
                 "Ghost": 5,
                 "Sorceror": 6,
                 "Minotaur": 7,
                 "HellRat": 8,
                 "EtherBat": 10,
                 "Bones": 12,
                 "RotWorm": 14,
                 "HungryGhost": 16,
                 "Liche": 18,
                 "Megataur": 20,
                 "Balrog": 28 }
sortedHazardRatings = sorted(hazardRatings.iteritems(), key=operator.itemgetter(1))

class Mobile(object):
    def __init__(self, x, y, ID=None, hp=1, str=1, agi=1, con=1, mag=1, wil=1):
        global curID
        
        self.maxHealth = hp
        self.health = self.maxHealth
        self.str = str
        self.agi = agi
        self.con = con
        self.mag = mag
        self.wil = wil
        
        # TODO: This is an ugly hack
        self.formHealth = self.health
        self.formStr = self.str
        self.formAgi = self.agi
        self.formCon = self.con
        self.formMag = self.mag
        self.formWil = self.wil
        
        self.ID = ID
        
        if self.ID == None:
            self.ID = curID
            curID += 1
        
        self.x = x
        self.y = y
        
        self.hasLos = False
        self.aggressive = False
        self.hazard = 0
        
        self.world = None
        self.img = None
        self.ai = None
        
        self.conditions = []
        self.onDeath = None
        self.dead = False
        
        self.baseSkills = []
        self.skills = []
        
        self.poisonStrength = 0
        self.poisonSource = None
        self.poisonedTime = 0
        self.poisonDuration = 0
        
        self.fearStrength = 0
        self.fearTime = 0
        self.fearDuration = 0
        
        self.disposed = False
        self.updated = False
        
        self.saveData = { "version": 0, \
                         "name": self.getName() }
        
    def save(self):
        # auto save integral types
        for attr, value in self.__dict__.items():
            if value == None or isinstance(value, (bool, basestring, float, int, long)):
                self.saveData[attr] = value
                
        self.saveData["conditions"] = self.conditions
        if self.poisonSource != None:
            self.saveData["poisonSource"] = self.poisonSource.ID
                
    def load(self, data):
        self.saveData = data
        
        # auto load integral types
        for attr, value in self.saveData.items():
            if value == None or isinstance(value, (bool, basestring, float, int, long)):
                self.__dict__[attr] = value
                
        self.conditions = self.saveData["conditions"]
        if self.saveData["poisonSource"] != None:
            self.poisonSource = self.world().getMobileByID(self.saveData["poisonSource"])
            
        if self.aggressive:
            self.aggro()
                
    def fearDebuff(self, strength):
        self.str -= strength
        self.agi -= strength
        self.con -= strength
        self.mag -= strength
        self.wil -= strength
        
    def setFearStrength(self, strength):
        if self.hasCondition(Condition.FEAR):
            self.fearDebuff(-self.fearStrength)
        
        self.fearStrength = strength
        self.fearDebuff(self.fearStrength)
                
    def heal(self, amount, showMsg=True):
        self.health += amount
        if self.health > self.maxHealth:
            self.health = self.maxHealth
        
        if showMsg:
            name = engine.getName(self)
            if name == "You":
                engine.log("You were healed for %s points" % amount)
            else:
                engine.log("%s was healed for %s points" % (name, amount))
        
    def addSkill(self, skill, base=True):
        if not skill in self.skills:
            self.skills.append(skill)
        
        if base:
            if not skill in self.baseSkills:
                self.baseSkills.append(skill)
        
    def hasCondition(self, condition):
        return condition in self.conditions
    
    def moveTo(self, x, y):
        res = self.checkCanMove(self.world().getTile(x, y))
        if res == None:
            res = self.world().checkCanMove(x, y)
            if res == None:
                oldPos = (self.x, self.y)
                self.x = x
                self.y = y
                self.world().mobileMoved(self, oldPos, (self.x, self.y))
            return res
        else:
            return res
        
    def move(self, direction):
        newPos = [self.x, self.y]
        
        if direction == Direction.DOWN:
            newPos[1] += 1
        elif direction == Direction.UP:
            newPos[1] -= 1
        elif direction == Direction.RIGHT:
            newPos[0] += 1
        elif direction == Direction.LEFT:
            newPos[0] -= 1
        elif direction == Direction.UPLEFT:
            newPos[0] -= 1
            newPos[1] -= 1
        elif direction == Direction.UPRIGHT:
            newPos[0] += 1
            newPos[1] -= 1
        elif direction == Direction.DOWNLEFT:
            newPos[0] -= 1
            newPos[1] += 1
        elif direction == Direction.DOWNRIGHT:
            newPos[0] += 1
            newPos[1] += 1
        
        return self.moveTo(newPos[0], newPos[1])
        
    def checkCanMove(self, tile):
        if tile.type == TileType.FLOOR or tile.type == TileType.STAIRSDOWN or tile.type == TileType.STAIRSUP:
            return None
        else:
            return tile
            
    def setWorld(self, world):
        self.world = weakref.ref(world)
        
    def removeWorld(self):
        self.world = None
        
    def removeCondition(self, condition):
        self.conditions.remove(condition)
        
    def update(self):
        self.updated = True
        
        if self.ai != None:
            return self.ai.tick()
        
    def updateStatus(self):
        regen = self.hasSkill("Regeneration")
        if regen != None:
            self.updated = True
            regen.resolve(self, self)
        
        if self.hasCondition(Condition.POISON):
            self.updated = True
            self.poisonedTime += 1
            if self.poisonedTime >= self.poisonDuration:
                self.removeCondition(Condition.POISON)
                self.poisonSource = None
            else:
                self.inflictDamage(self.poisonStrength, self.poisonSource)
        
        if self.hasCondition(Condition.FEAR):
            self.updated = True
            self.fearTime += 1
            if self.fearTime >= self.fearDuration:
                self.removeCondition(Condition.FEAR)
                self.fearDebuff(-self.fearStrength)
        
        if self.updated:
            self.save()
        
        self.updated = False
                
    def getPosition(self):
        return (self.x, self.y)
    
    def aggro(self):
        self.aggressive = True
        if not isinstance(self.ai, ai.Aggressive):
            self.ai = ai.Aggressive(self, engine.roomSize * 2)
        self.ai.aggro = True
    
    def inflictDamage(self, dmg, source):
        if self.world == None:
            print "inflicted damage on a being with no world"
            return   

        if source == engine.player.mobile:
            self.aggro()
            
        skill = self.hasSkill("Ethereal")
        if skill != None:
            if skill.resolve(self, self).success:
                dmg = 0
        
        self.health -= dmg
        if self.health <= 0:
            skill = self.hasSkill("Reanimate")
            if skill != None:
                res = skill.resolve(self, self)
                if res.sound != None:
                    engine.playSound(content.getSound(res.sound))

        if self.health <= 0:
            self.setDead(source)
            
    def cleanUp(self):
        self.world().removeMobile(self)
        self.removeWorld()
        self.disposed = True
        
    def setDead(self, source):
        self.dead = True
        if self.world == None and self.disposed:
            print "Mobile was set to dead, but world was None"
        else:
            self.world().mobileDied(self, source)
        if not self.dead:
            return
        self.cleanUp()
        
    def hasSkill(self, name):
        for skill in self.skills:
            if skill.name == name:
                return skill
        return None
                        
    def getName(self):
        return self.__class__.__name__

class Imp(Mobile):
    masterBonus = (0, 0, 0, 0, 0, 0)
    
    def __init__(self, x, y, ID=None):
        Mobile.__init__(self, x, y, ID, \
            hp=20, str=3, agi=4, con=1, mag=2, wil=2)
        
        self.img = content.getImage("imp")
        
        self.health = 1
        
class PlagueRat(Mobile):
    masterBonus = (0, 1, 1, 1, 0, 0)
    
    def __init__(self, x, y, ID=None):
        Mobile.__init__(self, x, y, ID, \
            hp=5, str=2, agi=6, con=1, mag=1, wil=1)
        
        self.hazard = hazardRatings["PlagueRat"]
        self.img = content.getImage("rat")
        self.ai = ai.Passive(self)
        
        self.addSkill(skills.Envenom())
           
class HellRat(Mobile):
    masterBonus = (0, 0, 1, 0, 1, 0)
    
    def __init__(self, x, y, ID=None):
        Mobile.__init__(self, x, y, ID, \
            hp=28, str=26, agi=31, con=29, mag=20, wil=20)
        
        self.hazard = hazardRatings["HellRat"]
        self.img = content.getImage("hellrat")
        self.ai = ai.Aggressive(self, scanDist=5, detectionChance=0.33)
        
        self.addSkill(skills.Envenom())
        self.addSkill(skills.Fireball())
               
class VampireBat(Mobile):
    masterBonus = (0, 0, 1, 0, 0, 1)
    
    def __init__(self, x, y, ID=None):
        Mobile.__init__(self, x, y, ID, \
            hp=6, str=3, agi=7, con=1, mag=3, wil=6)
        
        self.hazard = hazardRatings["VampireBat"]
        self.img = content.getImage("bat")
        self.ai = ai.Passive(self)
        
        self.addSkill(skills.LifeLeech())
          
class EtherBat(Mobile):
    masterBonus = (0, 0, 1, 0, 0, 1)
    
    def __init__(self, x, y, ID=None):
        Mobile.__init__(self, x, y, ID, \
            hp=30, str=33, agi=40, con=32, mag=25, wil=25)
        
        self.hazard = hazardRatings["EtherBat"]
        self.img = content.getImage("etherbat")
        self.ai = ai.Aggressive(self, detectionChance=0.5)
        
        self.addSkill(skills.LifeLeech())
        self.addSkill(skills.MagicMissile())
        self.addSkill(skills.Ethereal())
         
class Skeleton(Mobile):
    masterBonus = (3, 0, 0, 0, 0, 0)
    
    def __init__(self, x, y, ID=None):
        Mobile.__init__(self, x, y, ID, \
            hp=10, str=6, agi=6, con=4, mag=2, wil=2)
        
        self.hazard = hazardRatings["Skeleton"]
        self.img = content.getImage("skeleton")
        self.ai = ai.Aggressive(self, scanDist=6, detectionChance=0.25)
        
        self.addSkill(skills.Reanimate())
               
class Bones(Mobile):
    masterBonus = (0, 1, 0, 0, 0, 0)
    
    def __init__(self, x, y, ID=None):
        Mobile.__init__(self, x, y, ID, \
            hp=38, str=48, agi=46, con=34, mag=15, wil=26)
        
        self.hazard = hazardRatings["Bones"]
        self.img = content.getImage("bones")
        self.ai = ai.Mage(self, detectionChance=0.5)
        
        self.addSkill(skills.Reanimate())
        self.addSkill(skills.Charge())
        self.addSkill(skills.Regeneration())
        self.addSkill(skills.Chop())

class Mortipede(Mobile):
    masterBonus = (0, 0, 0, 0, 0, 1)
    
    def __init__(self, x, y, ID=None):
        Mobile.__init__(self, x, y, ID, \
            hp=12, str=8, agi=8, con=6, mag=6, wil=9)
        
        self.hazard = hazardRatings["Mortipede"]
        self.img = content.getImage("mortipede")
        self.ai = ai.Aggressive(self, detectionChance=0.33)
        
        self.addSkill(skills.Envenom())
        self.addSkill(skills.Fear())
               
class RotWorm(Mobile):
    masterBonus = (3, 0, 0, 1, 0, 0)
    
    def __init__(self, x, y, ID=None):
        Mobile.__init__(self, x, y, ID, \
            hp=36, str=44, agi=49, con=44, mag=22, wil=29)
        
        self.hazard = hazardRatings["RotWorm"]
        self.img = content.getImage("rotworm")
        self.ai = ai.Aggressive(self, detectionChance=0.6)
        
        self.addSkill(skills.Envenom())
        self.addSkill(skills.Fear())
        self.addSkill(skills.Reanimate())
        self.addSkill(skills.Charge())

class Ghost(Mobile):
    masterBonus = (0, 1, 0, 0, 0, 0)
    
    def __init__(self, x, y, ID=None):
        Mobile.__init__(self, x, y, ID, \
            hp=20, str=15, agi=13, con=12, mag=4, wil=4)
        
        self.hazard = hazardRatings["Ghost"]
        self.img = content.getImage("ghost")
        self.ai = ai.Aggressive(self, detectionChance=0.45)
        
        self.addSkill(skills.Reanimate())
        self.addSkill(skills.Ethereal())
               
class HungryGhost(Mobile):
    masterBonus = (0, 1, 0, 1, 0, 0)
    
    def __init__(self, x, y, ID=None):
        Mobile.__init__(self, x, y, ID, \
            hp=45, str=54, agi=50, con=52, mag=18, wil=24)
        
        self.hazard = hazardRatings["HungryGhost"]
        self.img = content.getImage("hungryghost")
        self.ai = ai.Aggressive(self, detectionChance=0.55)
        
        self.addSkill(skills.Reanimate())
        self.addSkill(skills.Ethereal())
        self.addSkill(skills.Regeneration())
        self.addSkill(skills.Fear())
               
class Sorceror(Mobile):
    masterBonus = (0, 0, 0, 0, 2, 0)
    
    def __init__(self, x, y, ID=None):
        Mobile.__init__(self, x, y, ID, \
            hp=16, str=6, agi=9, con=7, mag=14, wil=13)
        
        self.hazard = hazardRatings["Sorceror"]
        self.img = content.getImage("sorceror")
        self.ai = ai.Mage(self, detectionChance=0.6)
        
        self.addSkill(skills.LifeLeech())
        self.addSkill(skills.Fireball())
        self.addSkill(skills.MagicMissile())

class Liche(Mobile):
    masterBonus = (0, 0, 0, 0, 2, 0)
    
    def __init__(self, x, y, ID=None):
        Mobile.__init__(self, x, y, ID, \
            hp=39, str=19, agi=22, con=36, mag=46, wil=40)
        
        self.hazard = hazardRatings["Liche"]
        self.img = content.getImage("liche")
        self.ai = ai.Mage(self, detectionChance=0.75)
        
        self.addSkill(skills.LifeLeech())
        self.addSkill(skills.Fireball())
        self.addSkill(skills.MagicMissile())
        self.addSkill(skills.Ethereal())
        self.addSkill(skills.Reanimate())
        self.addSkill(skills.Regeneration())
        self.addSkill(skills.Fear())

class Minotaur(Mobile):
    masterBonus = (0, 1, 0, 1, 0, 0)
    
    def __init__(self, x, y, ID=None):
        Mobile.__init__(self, x, y, ID, \
            hp=30, str=24, agi=18, con=23, mag=8, wil=13)
        
        self.hazard = hazardRatings["Minotaur"]
        self.img = content.getImage("minotaur")
        self.ai = ai.Mage(self, detectionChance=0.5)
        
        self.addSkill(skills.Charge())
        self.addSkill(skills.Regeneration())
   
class Megataur(Mobile):
    masterBonus = (3, 0, 0, 1, 0, 0)
    
    def __init__(self, x, y, ID=None):
        Mobile.__init__(self, x, y, ID, \
            hp=59, str=72, agi=60, con=75, mag=27, wil=52)
        
        self.hazard = hazardRatings["Megataur"]
        self.img = content.getImage("megataur")
        self.ai = ai.Mage(self, detectionChance=0.66)
        
        self.addSkill(skills.Charge())
        self.addSkill(skills.Chop())
        self.addSkill(skills.Regeneration())
        self.addSkill(skills.Fear())
        
class Balrog(Mobile):
    masterBonus = (9, 3, 3, 3, 3, 3)
    
    def __init__(self, x, y, ID=None):
        Mobile.__init__(self, x, y, ID, \
            hp=95, str=92, agi=74, con=83, mag=66, wil=68)
        
        self.hazard = hazardRatings["Balrog"]
        self.img = content.getImage("balrog")
        self.ai = ai.Mage(self, scanDist=8, detectionChance=2.0)
        
        self.addSkill(skills.Charge())
        self.addSkill(skills.Chop())
        self.addSkill(skills.Regeneration())
        self.addSkill(skills.Envenom())
        self.addSkill(skills.Fear())
        self.addSkill(skills.LifeLeech())
        self.addSkill(skills.Fireball())
        self.addSkill(skills.MagicMissile())
        self.addSkill(skills.Ethereal())
        self.addSkill(skills.Reanimate())
        
def loadFromData(data, world):
    name = data["name"]
    mob = spawnByName(name, 0, 0, data["ID"])
    mob.setWorld(world)
    mob.load(data)
    return mob

def spawnByName(name, x, y, ID=None): # TODO: Replace with globals()[name]
    if name == "Imp":
        return Imp(x, y, ID)
    if name == "PlagueRat":
        return PlagueRat(x, y, ID)
    elif name == "Skeleton":
        return Skeleton(x, y, ID)
    elif name == "VampireBat":
        return VampireBat(x, y, ID)
    elif name == "Mortipede":
        return Mortipede(x, y, ID)
    elif name == "Ghost":
        return Ghost(x, y, ID)
    elif name == "Sorceror":
        return Sorceror(x, y, ID)
    elif name == "Minotaur":
        return Minotaur(x, y, ID)
    elif name == "HellRat":
        return HellRat(x, y, ID)
    elif name == "Bones":
        return Bones(x, y, ID)
    elif name == "EtherBat":
        return EtherBat(x, y, ID)
    elif name == "RotWorm":
        return RotWorm(x, y, ID)
    elif name == "HungryGhost":
        return HungryGhost(x, y, ID)
    elif name == "Liche":
        return Liche(x, y, ID)
    elif name == "Megataur":
        return Megataur(x, y, ID)
    elif name == "Balrog":
        return Balrog(x, y, ID)
    
if __name__ == "__main__":
    import pygame
    pygame.mixer.pre_init(44100, -16, 4, 2048)
    pygame.init()
    content.init()
    lastMob = Imp(0, 0)
    curStat = 0
    mobs = []
    
    for item in sortedHazardRatings:
        mob = spawnByName(item[0], 0, 0)
        curStat = int(round((hazardRatings[item[0]] - 1) * 0.5)) + curStat
        print mob.getName(), curStat
        mob.health = curStat
        mob.str = curStat
        mob.agi = curStat
        mob.con = curStat
        mob.mag = curStat
        mob.wil = curStat
        mobs.append(mob)
        
    requiredStats = {"health": 0, "str": 0, "agi": 0, "con": 0, "mag": 0, "wil": 0}
    
    for mob in mobs:
        requiredStats["health"] = max(max(mob.health - lastMob.health, 0), requiredStats["health"])
        requiredStats["str"] = max(max(mob.str - lastMob.str, 0), requiredStats["str"])
        requiredStats["agi"] = max(max(mob.agi - lastMob.agi, 0), requiredStats["agi"])
        requiredStats["con"] = max(max(mob.con - lastMob.con, 0), requiredStats["con"])
        requiredStats["mag"] = max(max(mob.mag - lastMob.mag, 0), requiredStats["mag"])
        requiredStats["wil"] = max(max(mob.wil - lastMob.wil, 0), requiredStats["wil"])
        
        print mob.getName(), requiredStats
        
        mob.health = mob.health + requiredStats["health"]
        mob.str = mob.str + requiredStats["str"]
        mob.agi = mob.agi + requiredStats["agi"]
        mob.con = mob.con + requiredStats["con"]
        mob.mag = mob.mag + requiredStats["mag"]
        mob.wil = mob.wil + requiredStats["wil"]
        
        #print mob.health, mob.str, mob.agi, mob.con, mob.mag, mob.wil
        
        lastMob = mob