import random

from enum import *
import mathhelper

class DynamicMusicEvent(object):
    def __init__(self, eventType, timeOut):
        self.type = eventType
        self.timeOut = timeOut * 1000
        self.elapsed = 0
        self.done = False
        
    def update(self, elapsed):
        self.elapsed += elapsed
        if self.elapsed >= self.timeOut:
            self.done = True
        
class EventDamage(DynamicMusicEvent):
    def __init__(self, timeOut=15):
        DynamicMusicEvent.__init__(self, MusicEvent.DAMAGED, timeOut)

class MusicHandler(object):
    def __init__(self, channels):
        self.channels = channels
        self.events = []
        self.music = []
        
        self.fading = False
        self.fadeTime = 0
        self.fadeInDefault = 500
        self.fadeOutDefault = 3000
        self.fadeTimeOut = self.fadeInDefault
        self.fadeFrom = 0.0
        self.fadeTarget = 1.0
        self.curFadeVolume = 0.0
        
        self.masterVolume = 1.0
        self.maxVolume = 1.0
        
    def addEvent(self, event):
        for i in xrange(len(self.events)):
            ev = self.events[i]
            if ev.type == event.type:
                if event.timeOut > ev.timeOut - ev.elapsed:
                    self.events[i] = event
                    return
        self.events.append(event)                
        
    def addMusic(self, music):
        if not music in self.music:
            self.music.append(music)
            
    def playRandom(self):
        music = self.music[random.randint(0, len(self.music)-1)]
        self.playMusic(music[0], 0, 1.0)
        self.playMusic(music[1], 1, self.curFadeVolume)
        
    def playMusic(self, sound, channel, volume):
        if sound.checkThread():
            self.channels[channel].play(sound.snd)
            self.channels[channel].set_volume(volume * self.maxVolume * self.masterVolume)
        
    def update(self, elapsed):
        if len(self.music) == 0:
            return
        
        if self.channels[0].get_sound() == None:
            self.playRandom()
        
        fadeOut = True
        
        for i in reversed(xrange(len(self.events))):
            event = self.events[i]
            if event.type == MusicEvent.DAMAGED:
                fadeOut = False
                
            event.update(elapsed)
            if event.done:
                del self.events[i]
                
        if fadeOut:
            self.fadeOut()
        else:
            self.fadeIn()
        
        if self.fading:
            self.fadeTime += elapsed
            
            factor = self.fadeTime / self.fadeTimeOut
            self.curFadeVolume = mathhelper.lerp(self.fadeFrom, self.fadeTarget, factor)
            
            self.channels[1].set_volume(self.curFadeVolume * self.maxVolume * self.masterVolume)
            
            if self.fadeTime >= self.fadeTimeOut:
                self.fading = False
                
    def fadeIn(self):
        if self.curFadeVolume >= 1.0:
            self.fading = False
            return
        if self.fading and self.fadeTarget == 1.0:
            return
        self.fadeTo(1.0, self.fadeInDefault)
        
    def fadeOut(self):
        if self.curFadeVolume <= 0.0:
            self.fading = False
            return
        if self.fading and self.fadeTarget == 0.0:
            return
        self.fadeTo(0.0, self.fadeOutDefault)
        
    def fadeTo(self, volume, fadeTime):
        self.fadeTime = 0
        self.fading = True
    
        self.fadeFrom = self.curFadeVolume
        self.fadeTarget = volume
        self.fadeTimeOut = abs(self.fadeFrom - volume) * fadeTime
        
    def setMaxVolume(self, volume):
        self.maxVolume = volume
        self.channels[0].set_volume(self.maxVolume * self.masterVolume)
        self.channels[1].set_volume(self.curFadeVolume * self.maxVolume * self.masterVolume)
        
    def setMasterVolume(self, volume):
        self.masterVolume = volume
        self.channels[0].set_volume(self.maxVolume * self.masterVolume)
        self.channels[1].set_volume(self.curFadeVolume * self.maxVolume * self.masterVolume)
