import heapq

class Cell(object):
    def __init__(self, x, y, reachable):
        """
        Initialize new cell

        @param x cell x coordinate
        @param y cell y coordinate
        @param reachable is cell reachable? not a wall?
        """
        self.reachable = reachable
        self.x = x
        self.y = y
        self.parent = None
        self.g = 0
        self.h = 0
        self.f = 0

class AStar(object):
    def __init__(self, start, end, getCellFunc, maxSteps=100, maxDist=16, blockedCells=[]):
        self.op = []
        heapq.heapify(self.op)
        self.cl = set()
        self.getCellFunc = getCellFunc
        self.maxSteps = maxSteps
        self.maxDist = maxDist
        self.start = None
        self.cellCache = {}
        self.start = self.get_cell(start[0], start[1], True)
        self.end = self.get_cell(end[0], end[1], True)
        
        for coords in blockedCells:
            self.cellCache[(coords[0], coords[1])] = Cell(coords[0], coords[1], False)

    def get_heuristic(self, cell):
        """
        Compute the heuristic value H for a cell: distance between
        this cell and the ending cell multiply by 10.
    
        @param cell
        @returns heuristic value H
        """
        return 10 * (abs(cell.x - self.end.x) + abs(cell.y - self.end.y))
    
    def get_cell(self, x, y, homeCell=False):
        """
        Returns a cell from the cells list
    
        @param x cell x coordinate
        @param y cell y coordinate
        @returns cell
        """
        if (x, y) in self.cellCache:
            return self.cellCache[(x, y)]
        
        if not homeCell:
            if abs(x - self.start.x) > self.maxDist or abs(y - self.start.y) > self.maxDist:
                return None
            cell = self.getCellFunc(x, y)
        else:
            cell = self.getCellFunc(x, y, True)
        
        self.cellCache[(x, y)] = cell
        return cell

    def get_adjacent_cells(self, cell):
        """
        Returns adjacent cells to a cell. Clockwise starting
        from the one on the right.
    
        @param cell get adjacent cells for this cell
        @returns adjacent cells list 
        """
        cells = []
        
        self.addCell(cell.x+1, cell.y+1, cells)
        self.addCell(cell.x, cell.y+1, cells)
        self.addCell(cell.x-1, cell.y+1, cells)
        self.addCell(cell.x+1, cell.y, cells)
        self.addCell(cell.x-1, cell.y, cells)
        self.addCell(cell.x+1, cell.y-1, cells)
        self.addCell(cell.x, cell.y-1, cells)
        self.addCell(cell.x-1, cell.y-1, cells)
        
        return cells
    
    def addCell(self, x, y, l):
        cell = self.get_cell(x, y)
        if cell != None:
            l.append(cell)

    def update_cell(self, adj, cell):
        """
        Update adjacent cell
    
        @param adj adjacent cell to current cell
        @param cell current cell being processed
        """
        adj.g = cell.g + 10
        adj.h = self.get_heuristic(adj)
        adj.parent = cell
        adj.f = adj.h + adj.g
        
    def tracePath(self):
        cell = self.end
        path = []
        while cell is not self.start:
            path.append((cell.x, cell.y))
            cell = cell.parent
        return path[::-1]

    def process(self):
        if self.start == None or self.end == None:
            return None
        
        # add starting cell to open heap queue
        heapq.heappush(self.op, (self.start.f, self.start))
        step = 0
        while len(self.op) and step < self.maxSteps:
            # pop cell from heap queue 
            f, cell = heapq.heappop(self.op)
            # add cell to closed list so we don't process it twice
            self.cl.add(cell)
            # if ending cell, display found path
            if cell is self.end:
                return self.tracePath()
                break
            # get adjacent cells for cell
            adj_cells = self.get_adjacent_cells(cell)
            for c in adj_cells:
                if c.reachable and c not in self.cl:
                    if (c.f, c) in self.op:
                        # if adj cell in open list, check if current path is
                        # better than the one previously found for this adj
                        # cell.
                        if c.g > cell.g + 10:
                            self.update_cell(c, cell)
                    else:
                        self.update_cell(c, cell)
                        # add adj cell to open list
                        heapq.heappush(self.op, (c.f, c))

            step += 1
                        
        return None
