import math

import pygame
from pygame import Rect

from enum import *

import content
import engine
import maps
import mathhelper

class Camera(object):
    def __init__(self):
        self.x = 0
        self.y = 0
        self.topLeft = (0, 0)
        self.screenTilesX = 0
        self.screenTilesY = 0
        self.target = None
        
    def update(self, screen):
        if self.target != None:
            self.x = self.target.x
            self.y = self.target.y
        
        self.screenTilesX = int(math.ceil(1.0 * screen.get_width() / engine.tileSize))
        self.screenTilesY = int(math.ceil(1.0 * screen.get_height() / engine.tileSize))
        
        self.topLeft = (self.x - self.screenTilesX / 2, self.y - self.screenTilesY / 2)
    
    def translate(self, x, y):
        return ((x - self.topLeft[0]) * engine.tileSize, (y - self.topLeft[1]) * engine.tileSize)

class Renderable(object):
    def __init__(self):
        self.isDisposed = False
    
    def render(self, screen):
        pass
    
    def cleanUp(self):
        self.isDisposed = True

class Renderer(object):
    def __init__(self, scale):
        self.scale = scale
        self.buffer = pygame.Surface((engine.screenWidth, engine.screenHeight))
        self.screen = pygame.display.set_mode((self.buffer.get_width() * self.scale, self.buffer.get_height() * self.scale))
        self.renderables = []
        
    def resize(self, scale):
        self.scale = scale
        self.screen = pygame.display.set_mode((self.buffer.get_width() * self.scale, self.buffer.get_height() * self.scale))

    def render(self):
        self.buffer.fill((0, 0, 0))
        
        for i in reversed(xrange(len(self.renderables))):
            renderable = self.renderables[i]
            
            if renderable.isDisposed:
                del self.renderables[i]
                
        for renderable in self.renderables:
            renderable.render(self.buffer)
        pygame.transform.scale(self.buffer, (self.screen.get_width(), self.screen.get_height()), self.screen)
        pygame.display.flip()

class LevelRenderer(Renderable):
    def __init__(self, level):
        Renderable.__init__(self)
        self.level = level # TODO: Make weakref
        self.camera = Camera()
        self.showAll = False
        
    def drawImage(self, screen, x, y, img, translate=True, blend=False):
        if translate:
            p = self.camera.translate(x, y)
        else:
            p = (x * engine.tileSize, y * engine.tileSize)
        a = img.get_alpha()
        if blend and a == None:
            img.set_alpha(63)
        elif not blend and a != None:
            img.set_alpha(None)
        screen.blit(img, p)

    def render(self, screen):
        Renderable.render(self, screen)
        
        floor = [content.getImage("floor0"), \
                 content.getImage("floor1"), \
                 content.getImage("floor2"), \
                 content.getImage("floor3")]
        wall = [content.getImage("wall0"), \
                content.getImage("wall1"), \
                content.getImage("wall2"), \
                content.getImage("wall3"), \
                content.getImage("wall4"), \
                content.getImage("wall5"), \
                content.getImage("wall6"), \
                content.getImage("wall7"), \
                content.getImage("wall8"), \
                content.getImage("wall9"), \
                content.getImage("wall10"), \
                content.getImage("wall11")]
        stairsUp = content.getImage("stairsup")
        stairsDown = content.getImage("stairsdown")
       
        self.camera.update(screen)
        
        rays = []
        #p0 = (self.camera.x, self.camera.y)
        
        left = self.camera.topLeft[0]
        right = left + self.camera.screenTilesX - 1
        top = self.camera.topLeft[1]
        bottom = top + self.camera.screenTilesY - 1
        
        """for x in xrange(self.camera.topLeft[0], self.camera.topLeft[0] + self.camera.screenTilesX):
            rays.append(mathhelper.Ray(p0, (x, top)))
            rays.append(mathhelper.Ray(p0, (x, bottom)))
            
        for y in xrange(self.camera.topLeft[1] + 1, self.camera.topLeft[1] + self.camera.screenTilesY - 1):
            rays.append(mathhelper.Ray(p0, (left, y)))
            rays.append(mathhelper.Ray(p0, (right, y)))
        
        litTiles = {}
        
        for ray in rays:
            while not ray.done:
                p = ray.step()
                x = p[0]
                y = p[1]
                
                tile = self.level.getTile(x, y)
                if tile != None:
                    litTiles[(x,y)] = tile
                    if tile.type == TileType.WALL:
                        break
                else:
                    pass
                    break
        
        for p, tile in litTiles.items():
            for y in xrange(p[1]-1, p[1]+2):
                for x in xrange(p[0]-1, p[0]+2):
                    if (x, y) in litTiles:
                        continue
                    
                    border = self.level.getTile(x, y)
                    
                    if border != None:
                        if border.type == TileType.WALL:
                            self.drawImage(screen, x, y, wall[border.style])
                        elif border.type == TileType.FLOOR:
                            self.drawImage(screen, x, y, floor[border.style])
                        elif border.type == TileType.STAIRSDOWN:
                            self.drawImage(screen, x, y, stairsDown)
                        elif border.type == TileType.STAIRSUP:
                            self.drawImage(screen, x, y, stairsUp)

            if tile.type == TileType.WALL:
                self.drawImage(screen, p[0], p[1], wall[tile.style])
            elif tile.type == TileType.FLOOR:
                self.drawImage(screen, p[0], p[1], floor[tile.style])
            elif tile.type == TileType.STAIRSDOWN:
                self.drawImage(screen, p[0], p[1], stairsDown)
            elif tile.type == TileType.STAIRSUP:
                self.drawImage(screen, p[0], p[1], stairsUp)"""
                
        tiles = []
        litTiles = {}
        p0 = (self.camera.x - self.camera.topLeft[0], self.camera.y - self.camera.topLeft[1])
                    
        for y in xrange(self.camera.topLeft[1], self.camera.topLeft[1] + self.camera.screenTilesY):
            tiles.append([])
            for x in xrange(self.camera.topLeft[0], self.camera.topLeft[0] + self.camera.screenTilesX):
                tiles[-1].append(self.level.getTile(x, y))
        
        for y in xrange(0, self.camera.screenTilesY):
            for x in xrange(0, self.camera.screenTilesX):
                tile = tiles[y][x]
                if tile != None:
                    ray = mathhelper.Ray((x, y), p0)
                    while not ray.done:
                        p = ray.step()
                        if tiles[p[1]][p[0]] == None or tiles[p[1]][p[0]].type == TileType.WALL:
                            break
                    
                    if ray.done:
                        for p in ray.points:
                            if not p in litTiles:
                                litTiles[p] = tiles[p[1]][p[0]]
                        
        drawn = set()
        
        for mob in self.level.losMobs:
            mob.hasLos = False
                        
        for p, tile in litTiles.items():
            for y in xrange(p[1]-1, p[1]+2):
                for x in xrange(p[0]-1, p[0]+2):
                    if (x, y) in litTiles or (x, y) in drawn or x < 0 or y < 0 or x >= self.camera.screenTilesX or y >= self.camera.screenTilesY:
                        continue
                    
                    if tiles[y][x] != None:
                        if not self.showAll:
                            if tiles[y][x].type == TileType.WALL:
                                self.drawImage(screen, x, y, wall[tiles[y][x].style], False)
                            elif tiles[y][x].type == TileType.FLOOR:
                                self.drawImage(screen, x, y, floor[tiles[y][x].style], False)
                            elif tiles[y][x].type == TileType.STAIRSDOWN:
                                self.drawImage(screen, x, y, stairsDown, False)
                            elif tiles[y][x].type == TileType.STAIRSUP:
                                self.drawImage(screen, x, y, stairsUp, False)
                        
                        tiles[y][x].visited = True
                        drawn.add((x, y))
                        self.drawMobs(screen, self.camera.topLeft[0] + x, self.camera.topLeft[1] + y)

            if not p in drawn:
                if not self.showAll:
                    if tile.type == TileType.WALL:
                        self.drawImage(screen, p[0], p[1], wall[tile.style], False)
                    elif tile.type == TileType.FLOOR:
                        self.drawImage(screen, p[0], p[1], floor[tile.style], False)
                    elif tile.type == TileType.STAIRSDOWN:
                        self.drawImage(screen, p[0], p[1], stairsDown, False)
                    elif tile.type == TileType.STAIRSUP:
                        self.drawImage(screen, p[0], p[1], stairsUp, False)
                
                tiles[p[1]][p[0]].visited = True
                drawn.add(p)
                self.drawMobs(screen, self.camera.topLeft[0] + p[0], self.camera.topLeft[1] + p[1])
                
        for y in xrange(self.camera.topLeft[1], self.camera.topLeft[1] + self.camera.screenTilesY):
            for x in xrange(self.camera.topLeft[0], self.camera.topLeft[0] + self.camera.screenTilesX):
                if not (x-self.camera.topLeft[0], y-self.camera.topLeft[1]) in drawn:
                    tile = self.level.getTile(x, y)
                    if tile != None and tile.visited:
                        if tile.type == TileType.WALL:
                            self.drawImage(screen, x, y, wall[tile.style], blend=True)
                        elif tile.type == TileType.FLOOR:
                            self.drawImage(screen, x, y, floor[tile.style], blend=True)
                        elif tile.type == TileType.STAIRSDOWN:
                            self.drawImage(screen, x, y, stairsDown, blend=True)
                        elif tile.type == TileType.STAIRSUP:
                            self.drawImage(screen, x, y, stairsUp, blend=True)
                                                            
        if self.showAll:
            for y in xrange(self.camera.topLeft[1], self.camera.topLeft[1] + self.camera.screenTilesY):
                for x in xrange(self.camera.topLeft[0], self.camera.topLeft[0] + self.camera.screenTilesX):
                    tile = self.level.getTile(x, y)
                    if tile != None:
                        if tile.type == TileType.WALL:
                            self.drawImage(screen, x, y, wall[tile.style])
                        elif tile.type == TileType.FLOOR:
                            self.drawImage(screen, x, y, floor[tile.style])
                        elif tile.type == TileType.STAIRSDOWN:
                            self.drawImage(screen, x, y, stairsDown)
                        elif tile.type == TileType.STAIRSUP:
                            self.drawImage(screen, x, y, stairsUp)
                    self.drawMobs(screen, x, y, False)

        """for mob in self.level.mobiles.values():
            relX = mob.x - self.camera.topLeft[0]
            relY = mob.y - self.camera.topLeft[1]
            
            if relX < 0 or relY < 0 or relX >= self.camera.screenTilesX or relY >= self.camera.screenTilesY:
                continue
            
            try:
                self.drawImage(screen, mob.x, mob.y, mob.img)
            except:
                pass"""

    def drawMobs(self, screen, x, y, setLos=True):
        for mob in self.level.mobilesAt(x, y):
            self.level.losMobs.append(mob)
            if setLos:
                mob.hasLos = True
            try:
                self.drawImage(screen, mob.x, mob.y, mob.img)
            except:
                pass