import random

from enum import *

import combat
import engine
import mobs
import mathhelper

class SkillResult(object):
    def __init__(self, success, sound):
        self.success = success
        self.sound = sound

class Skill(object):
    def __init__(self, name):
        self.name = name
        self.range = 1
        self.passive = False
        self.canTargetSelf = False
        self.swingSound = None
        self.hitSound = None
        self.isAttack = False
        self.dmgScalar = 1.0
        self.cost = 1
        self.desc = ""
        self.magical = True
        
    def resolve(self, source, target):
        return self.getResult(True, self.hitSound)
    
    def getResult(self, success, sound):
        return SkillResult(success, sound)
    
    def canTarget(self, target):
        if not self.canTargetSelf and target == engine.player.mobile:
            return False
        return True
    
    def getClassName(self):
        return self.__class__.__name__
        
class Possess(Skill):
    def __init__(self):
        Skill.__init__(self, "Possess")
        self.hitSound = "possess"
        self.desc = "Possess target after death"
        
    def resolve(self, source, target):
        if target.hasCondition(Condition.VESSEL):
            return self.getResult(False, "nope")
        combat.inflictCondition(source, target, Condition.VESSEL)
        
        return Skill.resolve(self, source, target)
        
class Depossess(Skill):
    def __init__(self):
        Skill.__init__(self, "Depossess")
        self.hitSound = "possess"
        self.range = 0
        self.canTargetSelf = True
        self.desc = "Return to imp form"
        self.cost = 0
        
    def canTarget(self, target):
        if engine.player.mobile.getName() == "Imp":
            return False
        return Skill.canTarget(self, target)
        
    def resolve(self, source, target):
        if engine.player.mobile.getName() == "Imp":
            return self.getResult(False, "nope")

        #engine.player.mobile.inflictDamage(engine.player.mobile.health, None)
        engine.player.mobile.setDead(True)
        return Skill.resolve(self, source, target)

class Envenom(Skill):
    def __init__(self):
        Skill.__init__(self, "Envenom")
        self.hitSound = "bump"
        self.swingSound = "envenom"
        self.isAttack = True
        self.dmgScalar = 0.5
        self.desc = "Deals damage equal to your level every turn"
        
    def resolve(self, source, target):
        dmg = combat.resolvePhysicalAttack(source, target, self.dmgScalar)
        if dmg <= 0:
            return self.getResult(False, None)
        
        combat.inflictCondition(source, target, Condition.POISON)
        if source.getName() == "Imp":
            target.poisonStrength = engine.player.level
        else:
            target.poisonStrength = mobs.hazardRatings[source.getName()]
        target.poisonedTime = 0
        target.poisonDuration = 5 + random.randint(0, 5)
        target.poisonSource = source
        return Skill.resolve(self, source, target)

class Fear(Skill):
    def __init__(self):
        Skill.__init__(self, "Fear")
        self.hitSound = "envenom"
        self.swingSound = "magic"
        self.isAttack = False
        self.desc = "Debuff to all stats, Wil based to-hit"
        self.range = 4
        self.cost = 4
        
    def resolve(self, source, target):
        wilDif = abs(source.wil - target.wil)
        tgrWil = max(target.wil, 1)
        srcWil = max(source.wil, 1)
        if source.wil > target.wil:
            toHitMod = float(wilDif) / tgrWil / 2.0
        elif source.agi < target.agi:
            toHitMod = -float(wilDif) / srcWil / 4.0
        else:
            toHitMod = 0
            
        toHit = mathhelper.clamp(0.5 + toHitMod, 0.2, 0.8)
        
        if not mathhelper.success(toHit):
            # TODO: add log
            return self.getResult(False, None)
        
        combat.inflictCondition(source, target, Condition.FEAR)
        if source.getName() == "Imp":
            target.setFearStrength(int(engine.player.level / 2))
        else:
            target.setFearStrength(int(mobs.hazardRatings[source.getName()] / 2))
        target.fearTime = 0
        target.fearDuration = 5 + random.randint(0, 5)
        return Skill.resolve(self, source, target)

class LifeLeech(Skill):
    def __init__(self):
        Skill.__init__(self, "Life Leech")
        self.hitSound = "bump"
        self.swingSound = "envenom"
        self.isAttack = True
        self.dmgScalar = 0.5
        self.magical = True
        self.desc = "Leech opponent's HP"
        self.cost = 3
        
    def resolve(self, source, target):
        dmg = combat.resolveMagicAttack(source, target, self.dmgScalar)
        source.heal(dmg)
        return Skill.resolve(self, source, target)
    
class Chop(Skill):
    def __init__(self):
        Skill.__init__(self, "Chop")
        self.hitSound = "bump"
        self.swingSound = "swing"
        self.isAttack = True
        self.dmgScalar = 3
        self.magical = False
        self.desc = "A devastating attack with reach"
        self.cost = 11
        self.range = 2
        
    def resolve(self, source, target):
        dmg = combat.resolveMagicAttack(source, target, self.dmgScalar)
        return Skill.resolve(self, source, target)
        
class Fireball(Skill):
    def __init__(self):
        Skill.__init__(self, "Fireball")
        self.hitSound = "fireball"
        self.swingSound = "magic"
        self.isAttack = True
        self.dmgScalar = 2
        self.magical = True
        self.desc = "Cast a fireball for massive damage"
        self.cost = 8
        self.range = 4
        
    def resolve(self, source, target):
        dmg = combat.resolveMagicAttack(source, target, self.dmgScalar)
        return Skill.resolve(self, source, target)
    
class MagicMissile(Skill):
    def __init__(self):
        Skill.__init__(self, "Magic Missile")
        self.hitSound = "bump"
        self.swingSound = "magic"
        self.isAttack = True
        self.dmgScalar = 0.33
        self.magical = True
        self.desc = "Cast three minor magic missiles"
        self.cost = 5
        self.range = 3
        
    def resolve(self, source, target):
        dmg = combat.resolveMagicAttack(source, target, self.dmgScalar)
        dmg = combat.resolveMagicAttack(source, target, self.dmgScalar)
        dmg = combat.resolveMagicAttack(source, target, self.dmgScalar)
        return Skill.resolve(self, source, target)
    
class Charge(Skill):
    def __init__(self):
        Skill.__init__(self, "Charge")
        self.hitSound = "bump"
        self.swingSound = "charge"
        self.isAttack = True
        self.dmgScalar = 1.5
        self.magical = False
        self.desc = "Charge your enemy across a distance"
        self.cost = 5
        self.range = 3
        
    def resolve(self, source, target):
        world = source.world()
        
        positions = []
        for y in xrange(-1, 2):
            for x in xrange(-1, 2):
                if source.checkCanMove(world.getTile(target.x + x, target.y + y)) == None and world.checkCanMove(target.x + x, target.y + y) == None:
                    positions.append((target.x + x, target.y + y))
        
        if len(positions) == 0:
            if source == engine.player.mobile:
                engine.log("Charge failed!")
            # TODO: add log
            return self.getResult(False, None)
        
        closestPos = positions[0]
        for coords in positions:
            if engine.dist(coords[0], coords[1], source.x, source.y) < engine.dist(closestPos[0], closestPos[1], source.x, source.y):
                closestPos = coords
                
        dmg = combat.resolvePhysicalAttack(source, target, self.dmgScalar)
        
        source.moveTo(closestPos[0], closestPos[1])
        return Skill.resolve(self, source, target)
    
class Regeneration(Skill):
    def __init__(self):
        Skill.__init__(self, "Regeneration")
        self.isAttack = False
        self.passive = True
        self.desc = "Regenerate HP every turn"
        self.cost = 0
        self.range = 0
        
    def resolve(self, source, target):
        source.heal(max(int(round(source.con / 20.0)), 1), False)
        return Skill.resolve(self, source, target)
    
class Ethereal(Skill):
    def __init__(self):
        Skill.__init__(self, "Ethereal")
        self.cost = 0
        self.passive = True
        self.desc = "10% chance to negate physical damage"
        
    def resolve(self, source, target):
        if mathhelper.success(0.1):
            name = engine.getName(source)
            engine.log("%s negated all damage!" % name)
            return Skill.resolve(self, source, target)
        return self.getResult(False, None)
        
class Reanimate(Skill):
    def __init__(self):
        Skill.__init__(self, "Reanimate")
        self.cost = 0
        self.passive = True
        self.desc = "20% chance to reanimate with 30% HP on death"
        
    def resolve(self, source, target):
        if mathhelper.success(0.2):
            name = engine.getName(source)
            engine.log("%s died but reanimated!" % name)
            source.dead = False
            source.health = max(int(round(source.maxHealth * 0.3)), 1)
            return Skill.resolve(self, source, target)
        return self.getResult(False, None)