import pygame
from pygame import Rect
from collections import OrderedDict

import render
import font
import content
import engine
import options
from enum import *

class UI(render.Renderable):
    def __init__(self, renderer):
        self.titleScreen = TitleScreen()
        self.mainMenu = MainMenu()
        self.skillPane = SkillPane()
        self.possessPane = PossessPane()
        self.statsPane = StatsPane()
        self.formPane = FormPane()
        self.bottomPane = BottomPane()
        self.textLog = TextLog()
        self.deathScreen = DeathScreen()
        self.options = OptionsMenu()
        
        self.options.isInGame = self.mainMenu.isInGame
        
        renderer.renderables.append(self.titleScreen)
        renderer.renderables.append(self.bottomPane)
        renderer.renderables.append(self.textLog)
        renderer.renderables.append(self.skillPane)
        renderer.renderables.append(self.statsPane)
        renderer.renderables.append(self.formPane)
        renderer.renderables.append(self.possessPane)
        renderer.renderables.append(self.deathScreen)
        renderer.renderables.append(self.mainMenu)
        renderer.renderables.append(self.options)

class UIWindow(render.Renderable):
    def __init__(self):
        render.Renderable.__init__(self)    
        self.visible = True
        self.lineY = 0
        self.lineX = 0
        self.lineHeight = 14

    def show(self):
        self.visible = True
        self.onShow()
        
    def onShow(self):
        pass
        
    def hide(self):
        self.visible = False
        self.onHide()
        
    def onHide(self):
        pass
    
    def setCursor(self, x, y):
        self.lineX = x
        self.lineY = y
    
    def printLine(self, screen, text, color=(0, 0, 0)):
        font.drawText(screen, str(text), self.lineX, self.lineY, color=color)
        self.lineY += self.lineHeight
        
class DeathScreen(UIWindow):
    def __init__(self):
        UIWindow.__init__(self)    
        self.visible = False
        
    def render(self, screen):
        if not self.visible:
            return
        
        screen.blit(content.getImage("spotlight"), (0, 0))
        gameOver = content.getImage("gameover")
        screen.blit(gameOver, ((engine.screenWidth - gameOver.get_width()) / 2, (engine.screenHeight - gameOver.get_height()) / 2))
           
class TextLog(UIWindow):
    def __init__(self):
        UIWindow.__init__(self)
        self.style = 6
        self.visible = False
        
    def switchStyle(self):
        self.style += 1
        if self.style > 6:
            self.style = 0
        
    def render(self, screen):
        if self.style == 0 or not self.visible:
            return
        
        pane = content.getImage("bottompane")
        bg = content.getImage("textlog")
        
        y = screen.get_height() - pane.get_height()

        if self.style < 6:
            lines = self.style
            
        if self.style == 6:
            screen.blit(bg, (0, 0))
        else:
            height = 4 + 13 * lines
            screen.blit(bg, (0, y - height), Rect(0, 0, bg.get_width(), height))
        
        y -= 15
        i = 0
        for line in engine.logText:
            if self.style < 6 and i == lines:
                break
            font.drawText(screen, line, 3, y, color=(255, 255, 255), background=False)
            y -= 13
            i += 1
       
class BottomPane(UIWindow):
    def __init__(self):
        UIWindow.__init__(self)
        self.floor = 1
        self.visible = False
        
    def render(self, screen):
        if not self.visible:
            return

        pane = content.getImage("bottompane")
        y = screen.get_height() - pane.get_height()
        screen.blit(pane, (0, screen.get_height() - pane.get_height()))
        
        x = 5
        y += 3
        font.drawText(screen, "Lvl: %s   HP: %s/%s   SP: %s/%s      %sF" % 
                      (engine.player.level, engine.player.mobile.health,
                       engine.player.mobile.maxHealth, int(engine.player.sp), engine.player.maxSp,
                       self.floor), x, y, True)
        
        x = pane.get_width() - 16
        y = screen.get_height() - pane.get_height()
        
        if engine.player.mobile.hasCondition(Condition.POISON):
            perc = 1.0 - float(engine.player.mobile.poisonedTime) / engine.player.mobile.poisonDuration
            self.drawStatusIcon(screen, content.getImage("poison"), x, y, perc)
            x -= 16
            
        if engine.player.mobile.hasCondition(Condition.FEAR):
            perc = 1.0 - float(engine.player.mobile.fearTime) / engine.player.mobile.fearDuration
            self.drawStatusIcon(screen, content.getImage("fear"), x, y, perc)
            x -= 16
            
    def drawStatusIcon(self, screen, img, x, y, perc):
        img.set_alpha(64)
        screen.blit(img, (x, y))
        img.set_alpha(None)
        
        height = int(round(img.get_height() * perc))
        width = img.get_width()
        padding = img.get_height() - height
        screen.blit(img, (x, y+padding), area=Rect(0, padding, width, height))

class TitleScreen(UIWindow):
    def __init__(self):
        UIWindow.__init__(self)
        self.visible = True
        
    def render(self, screen):
        if not self.visible:
            return

        screen.blit(content.getImage("tilebackground"), (0, 0))
        screen.blit(content.getImage("spotlight"), (0, 0))
        screen.blit(content.getImage("titlescreen"), (0, 0))

class MainMenu(UIWindow):
    def __init__(self):
        UIWindow.__init__(self)
        self.visible = False
        self.inGame = False
        self.cursorPos = 0
        self.positions = [(90, 112), (100, 141), (105, 170), (118, 199)]
        self.fromMode = 0
        
    def isInGame(self):
        return self.inGame
        
    def moveCursor(self, event, key, info):
        if info == "Down" or info == "Right":
            self.cursorPos += 1
        elif info == "Up" or info == "Left":
            self.cursorPos -= 1
        
        if self.cursorPos >= len(self.positions):
            self.cursorPos = 0
        if self.cursorPos < 0:
            self.cursorPos = len(self.positions) - 1
        
    def render(self, screen):
        if not self.visible:
            return
        
        if not self.inGame:
            screen.blit(content.getImage("tilebackground"), (0, 0))
        
        screen.blit(content.getImage("spotlight"), (0, 0))
        screen.blit(content.getImage("mainmenu"), (0, 0))
        screen.blit(content.getImage("bigcursor"), self.positions[self.cursorPos])
        
class PossessPane(UIWindow):
    def __init__(self):
        UIWindow.__init__(self)
        self.choice = True
        self.visible = False
        self.mob = None
        
    def moveCursor(self, event, key, info):
        self.choice = not self.choice
        
    def render(self, screen):
        if not self.visible:
            return
        
        pane = content.getImage("possesspane")
        top = (screen.get_height() - pane.get_height()) / 2
        left = 11
        screen.blit(content.getImage("possesspane"), (11, top))
        
        top += 4
        
        font.drawText(screen, "You have defeated %s" % self.mob.getName(), left, top+4, True, align=True, width=pane.get_width())
        font.drawText(screen, "Possess this vessel?", left, top+18, True, align = True, width=pane.get_width())
        
        x = left + pane.get_width() / 2
        y = top+46
        font.drawText(screen, "Yes", x - 32, y, True)
        font.drawText(screen, "No", x + 10, y, True)
        
        if self.choice:
            screen.blit(content.getImage("cursor"), (x - 32 - 18, y))
        else:
            screen.blit(content.getImage("cursor"), (x + 10 - 18, y))

class StatsPane(UIWindow):
    def __init__(self):
        UIWindow.__init__(self)
        self.cursorPos = 0
        self.visible = False
        
    def moveCursor(self, event, key, info):
        if info == "Down":
            self.cursorPos += 1
        elif info == "Up":
            self.cursorPos -= 1
        elif info == "Right":
            self.cursorPos += 1
        elif info == "Left":
            self.cursorPos -= 1
            
        while self.cursorPos >= 6:
            self.cursorPos -= 6
        while self.cursorPos < 0:
            self.cursorPos += 6
                
    def render(self, screen):
        if not self.visible:
            return
        
        pane = content.getImage("pane")
        screen.blit(pane, (11, 30))
        
        x = 24
        startY = 34
        y = startY
        i = 0
        
        player = engine.player
        mob = engine.player.mobile
        
        if engine.player.statPoints > 0:
            screen.blit(content.getImage("cursor"), (x - 18, y + self.lineHeight * (self.cursorPos + 1)))
            
        self.setCursor(x, y)
        
        self.printLine(screen, "Status")
        self.printLine(screen, "HP:")
        self.printLine(screen, "Str:")
        self.printLine(screen, "Agi:")
        self.printLine(screen, "Con:")
        self.printLine(screen, "Mag:")
        self.printLine(screen, "Wil:")
        self.printLine(screen, "")
        self.printLine(screen, "Level: %s" % (player.level))
        self.printLine(screen, "Exp: %s/%s" % (player.exp, player.expToLevel))
        self.printLine(screen, "Stat points: %s" % (player.statPoints))
        
        self.setCursor(x + 33, y + self.lineHeight)
        red = (255, 0, 0)
        color = red if player.health > player.mobile.formHealth else (0, 0, 0)
        self.printLine(screen, "%s / %s" % (player.mobile.health, player.mobile.maxHealth), color)
        color = red if player.str > player.mobile.formStr else (0, 0, 0)
        self.printLine(screen, player.mobile.str, color)
        color = red if player.agi > player.mobile.formAgi else (0, 0, 0)
        self.printLine(screen, player.mobile.agi, color)
        color = red if player.con > player.mobile.formCon else (0, 0, 0)
        self.printLine(screen, player.mobile.con, color)
        color = red if player.mag > player.mobile.formMag else (0, 0, 0)
        self.printLine(screen, player.mobile.mag, color)
        color = red if player.wil > player.mobile.formWil else (0, 0, 0)
        self.printLine(screen, player.mobile.wil, color)

        self.setCursor(x + 100, y)
        self.printLine(screen, "Base/Max")
        self.printLine(screen, "%s / %s" % (player.health, player.mobile.formHealth))
        self.printLine(screen, "%s / %s" % (player.str, player.mobile.formStr))
        self.printLine(screen, "%s / %s" % (player.agi, player.mobile.formAgi))
        self.printLine(screen, "%s / %s" % (player.con, player.mobile.formCon))
        self.printLine(screen, "%s / %s" % (player.mag, player.mobile.formMag))
        self.printLine(screen, "%s / %s" % (player.wil, player.mobile.formWil))
        
        self.setCursor(x + 175, y)
        self.printLine(screen, "Mastery")
        self.printLine(screen, "%s" % (player.masterHealth))
        self.printLine(screen, "%s" % (player.masterStr))
        self.printLine(screen, "%s" % (player.masterAgi))
        self.printLine(screen, "%s" % (player.masterCon))
        self.printLine(screen, "%s" % (player.masterMag))
        self.printLine(screen, "%s" % (player.masterWil))
        
        self.setCursor(x + 235, y)
        data = engine.player.getFormData()
        self.printLine(screen, "Overlvl")
        self.printLine(screen, "%s" % (data.health))
        self.printLine(screen, "%s" % (data.str))
        self.printLine(screen, "%s" % (data.agi))
        self.printLine(screen, "%s" % (data.con))
        self.printLine(screen, "%s" % (data.mag))
        self.printLine(screen, "%s" % (data.wil))
        
class FormPane(UIWindow):
    def __init__(self):
        UIWindow.__init__(self)
        self.cursorPos = 0
        self.visible = False
        
    def moveCursor(self, event, key, info):
        if info == "Down":
            self.cursorPos += 1
        elif info == "Up":
            self.cursorPos -= 1
        elif info == "Right":
            self.cursorPos += 1
        elif info == "Left":
            self.cursorPos -= 1
            
        while self.cursorPos >= 6:
            self.cursorPos -= 6
        while self.cursorPos < 0:
            self.cursorPos += 6
                
    def render(self, screen):
        if not self.visible:
            return
        
        pane = content.getImage("pane")
        paneX = 11
        screen.blit(pane, (paneX, 30))
        
        x = 24
        startY = 34
        y = startY
        i = 0
        
        self.setCursor(x, y)
        
        player = engine.player
        mob = engine.player.mobile
        
        mobName = mob.getName()
        skillExpToLevel = player.getSkillExpToLevel()
        skillExp = player.getSkillExp(mobName)
        formExpToLevel = player.getFormExpToLevel()
        formExp = player.getFormExp(mobName)
        formPoints = player.getFormPoints()
        
        formStr = "Current form: %s" % (mobName)
        if player.isFormMaster():
            formStr += " [MASTER]"
        
        self.printLine(screen, formStr)
        
        if len(player.getUnlearnedSkills()) > 0:
            self.printLine(screen, "Skill exp: %s / %s" % (skillExp, skillExpToLevel))
        else:
            self.printLine(screen, "Skill exp: - / -")
        
        if not player.isFormMaster():
            self.printLine(screen, "Form exp: - / -")
        else:
            self.printLine(screen, "Form exp: %s / %s" % (formExp, formExpToLevel))
        self.printLine(screen, "Overlevel points: %s" % (formPoints))
        self.printLine(screen, "")
        
        statsY = self.lineY
        self.printLine(screen, "HP:")
        self.printLine(screen, "Str:")
        self.printLine(screen, "Agi:")
        self.printLine(screen, "Con:")
        self.printLine(screen, "Mag:")
        self.printLine(screen, "Wil:")
        
        self.setCursor(x + 33, statsY)
        self.printLine(screen, player.mobile.maxHealth)
        self.printLine(screen, player.mobile.str)
        self.printLine(screen, player.mobile.agi)
        self.printLine(screen, player.mobile.con)
        self.printLine(screen, player.mobile.mag)
        self.printLine(screen, player.mobile.wil)
        
        self.setCursor(paneX + pane.get_width() / 2, y + self.lineHeight)
        self.printLine(screen, "Monster skills:")
        for skill in mob.baseSkills:
            self.printLine(screen, "%s" % (skill.name))
        
        if formPoints > 0:
            screen.blit(content.getImage("cursor"), (x - 18, statsY + self.lineHeight * self.cursorPos))

class SkillPane(UIWindow):
    def __init__(self):
        UIWindow.__init__(self)
        self.cursorPos = 0
        self.visible = False
        self.skillsPerColumn = 10
        
    def moveCursor(self, event, key, info):
        if info == "Down":
            self.cursorPos += 1
        elif info == "Up":
            self.cursorPos -= 1
        elif info == "Right":
            self.cursorPos += self.skillsPerColumn
        elif info == "Left":
            self.cursorPos -= self.skillsPerColumn
            
        while self.cursorPos >= len(self.getSkills()):
            self.cursorPos -= len(self.getSkills())
        while self.cursorPos < 0:
            self.cursorPos += len(self.getSkills())
            
    def getSelectedSkill(self):
        return self.getSkills()[self.cursorPos]
    
    def getSkills(self):
        skills = engine.player.skills[:] + engine.player.getUnlearnedSkills()
        return skills
     
    def render(self, screen):
        if not self.visible:
            return
        
        screen.blit(content.getImage("pane"), (11, 30))
        
        x = 24
        startY = 34
        y = startY
        i = 0
        
        skills = self.getSkills()
        
        if self.cursorPos >= len(skills):
            self.cursorPos = 0
        curSkill = skills[self.cursorPos]
        
        if curSkill.passive:
            info = "Passive skill"
        else:
            info = "Rng: %s, TrgtSelf: %s" % (curSkill.range, curSkill.canTargetSelf)
            if curSkill.isAttack:
                if curSkill.magical:
                    info += ", MagAtk"
                else:
                    info += ", PhysAtk"
                info += ", Dmg: %s%%" % (int(round(curSkill.dmgScalar * 100)))
        
        descY = startY + (self.skillsPerColumn * 14)
        font.drawText(screen, curSkill.desc, x - 6, descY, True)
        font.drawText(screen, info, x - 6, descY + 14, True)
        
        baseSkillLen = len(engine.player.skills)
        
        for skill in skills:
            if i >= baseSkillLen:
                prefix = "*"
            else:
                prefix = ""
            
            if skill.passive:
                font.drawText(screen, "%s%s" % (prefix, skill.name), x, y, True, color=(96, 96, 96))
            else:
                font.drawText(screen, "%s%s (%s)" % (prefix, skill.name, skill.cost), x, y, True, color=self.getSkillColor(skill))
                
            if i == self.cursorPos:
                screen.blit(content.getImage("cursor"), (x - 18, y))

            y += 14
            if i != 0 and (i+1) % self.skillsPerColumn == 0:
                x += 97
                y = startY

            i += 1
            
    def getSkillColor(self, skill):
        if skill.cost > int(engine.player.sp):
            return (255, 0, 0)
        else:
            return (0, 0, 0)

class OptionsMenu(UIWindow):
    def __init__(self):
        UIWindow.__init__(self)
        self.visible = False
        self.isInGame = None
        self.config = None
        self.cursorPos = 0
        self.viewPos = 0
        self.keyBinds = []
        self.maxRows = 11
        self.altPos = False
        self.colSize = 75
        self.colPadding = 7
        self.binding = False
        
        self.items = OrderedDict({})
        
    def apply(self):
        options = self.items.values()
        
        for option in options:
            if option.type != OptionType.HEADER and option.type != OptionType.BLANK:
                if option.type == OptionType.KEY:
                    self.config.set(option.cfgSection, option.cfgName, option.value, True, option.altValue)
                else:
                    self.config.set(option.cfgSection, option.cfgName, option.value)
                    
    def getVolume(self):
        master = 1
        sfx = 1
        music = 1
        for option in self.items.values():
            if isinstance(option, options.MasterVolume):
                master = option.value
            elif isinstance(option, options.SoundVolume):
                sfx = option.value
            elif isinstance(option, options.MusicVolume):
                music = option.value
                
        return (master, sfx, music)
    
    def reset(self):
        self.items.clear()
        
        self.addItem(options.MasterVolume())
        self.addItem(options.SoundVolume())
        self.addItem(options.MusicVolume())
        self.addItem(options.RememberTarget())
        self.addItem(options.MultiKeyDiagonals())
        self.addItem(options.AttackOnBump())
        self.addItem(options.AttackConfirms())
        self.addItem(options.Blank("1"))
        self.addItem(options.Header("[Key binds]"))
        
        for name in self.keyBinds:
            self.addItem(options.Key(name))
        
    def addItem(self, item):
        self.items[item.cfgName.lower()] = item
        
    def moveCursor(self, event, key, info):
        option = self.items.values()[self.cursorPos]
        hasAlt = not option.type != OptionType.KEY
        
        if info == "Down":
            self.cursorPos += 1
        elif info == "Up":
            self.cursorPos -= 1

        if option.type == OptionType.INT or option.type == OptionType.FLOAT:
            if info == "Left":
                option.stepDown()
                return
            elif info == "Right":
                option.stepUp()
                return 
        
        if self.cursorPos >= len(self.items):
            self.cursorPos = 0
        if self.cursorPos < 0:
            self.cursorPos = len(self.items) - 1
            
        if info == "Down":
            self.cursorToNextItem()
        elif info == "Up":
            self.cursorToNextItem(-1)
            
        if self.cursorPos < self.viewPos:
            self.viewPos = self.cursorPos
        elif self.cursorPos >= self.viewPos + self.maxRows:
            self.viewPos = self.cursorPos - self.maxRows + 1
        
        if not hasAlt:
            self.altPos = False
        elif info == "Left" or info == "Right":
            self.altPos = not self.altPos
                
    def cursorToNextItem(self, step=1):
        wrapped = False
        options = self.items.values()
        i = self.cursorPos
        while True:
            option = options[i]
            
            if option.type != OptionType.BLANK and option.type != OptionType.HEADER:
                self.cursorPos = i
                break
            i += step
            
            if i >= len(self.items) or i < 0:
                wrapped = True
                if i < 0:
                    i = len(self.items) - 1
                else:
                    i = 0
                
            if i == self.cursorPos and wrapped:
                break
    
    def onShow(self):
        self.reset()
        
        self.cursorPos = 0
        self.viewPos = 0
        self.cursorToNextItem()
        
        options = self.config.getOptions()
        for optionName in options:
            if optionName.lower() in self.items:
                item = self.items[optionName.lower()]
                
                if item.type == OptionType.KEY:
                    hasAlt = True
                else:
                    hasAlt = False
                    
                option = self.config.getOption(item.cfgSection, item.cfgName, hasAlt)
                
                if item.type != OptionType.STRING:
                    item.value = option.value
                    item.altValue = option.alt
                else:
                    item.value = option.rawValue
                    item.altValue = option.rawAlt
                    
    def unbind(self):
        option = self.items.values()[self.cursorPos]
        if option.type == OptionType.KEY:
            if self.altPos:
                option.altValue = None
            else:
                option.value = None
                
    def bindKey(self, key):
        option = self.items.values()[self.cursorPos]
        if option.type == OptionType.KEY:
            if self.altPos:
                option.altValue = key
            else:
                option.value = key
                
        self.binding = False
                            
    def confirm(self):
        option = self.items.values()[self.cursorPos]
        
        if option.type == OptionType.BOOL:
            option.toggle()
        if option.type == OptionType.KEY:
            self.binding = True
     
    def render(self, screen):
        if not self.visible:
            return
        
        pane = content.getImage("pane")
        
        if not self.isInGame():
            screen.blit(content.getImage("tilebackground"), (0, 0))
        screen.blit(content.getImage("spotlight"), (0, 0))
        screen.blit(pane, (11, 30))

        x = 24
        startY = 34
        y = startY
        
        font.drawText(screen, "Options", 11, y, True, align=True, width=pane.get_width())
        y += 14
        
        if self.config is None:
            return
        
        i = 0
        options = self.items.values()
        optionWidth = pane.get_width() - 11 - (x - 11)
        
        for optionIndex in xrange(self.viewPos, min(self.viewPos + self.maxRows, len(options))):
            option = options[optionIndex]
            
            font.drawText(screen, option.name, x, y, True)
            
            self.renderOption(screen, option, x, y, optionWidth)
            
            if optionIndex == self.cursorPos and option.type != OptionType.KEY:
                screen.blit(content.getImage("cursor"), (x - 18, y))
            elif optionIndex == self.cursorPos and not self.binding:
                altX = x + optionWidth - self.colSize * 2 - self.colPadding
                if self.altPos:
                    altX += self.colSize + self.colPadding
                screen.blit(content.getImage("cursor"), (altX - 18, y))

            y += 14
            i += 1
            
            if i == self.maxRows:
                break

    def renderOption(self, screen, option, x, y, width):
        if option.type == OptionType.BOOL:
            if option.value:
                check = content.getImage("checkon")
            else:
                check = content.getImage("checkoff")
                
            screen.blit(check, (x + width - check.get_width(), y))
            
        elif option.type == OptionType.INT:
            font.drawText(screen, str(option.value), x, y, True, Align.RIGHT, width)
        
        elif option.type == OptionType.KEY:
            if option.value != None:
                font.drawText(screen, pygame.key.name(option.value), x + width - self.colSize * 2 - self.colPadding, y, True, Align.CENTER, self.colSize)
            if option.altValue != None:
                font.drawText(screen, pygame.key.name(option.altValue), x + width - self.colSize, y, True, Align.CENTER, self.colSize)
